<?php

namespace App\Http\Controllers\Supplier_Controllers;

use App\Enums\MultiPurposeStatus;
use App\Enums\PaymentStatus;
use App\Enums\StockHistorySource;
use App\Enums\ReturnType;
use App\Enums\TransactionCode;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Function_Controllers\StockFunction;
use App\Http\Controllers\Function_Controllers\StockHistoryFunction;
use App\Models\Stock_Models\Stock;
use App\Models\Stock_Models\StockHistory;
use App\Models\Supplier_Models\PurchaseReturn;
use App\Models\Supplier_Models\Supplier;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseReturnController extends Controller
{

    private $stock_history_function;
    private $stock_function;
    public function __construct()
    {
        $this->stock_history_function = new StockHistoryFunction();
        $this->stock_function = new StockFunction();

    }

    public function index(Request $request)
    {

        try {
            $purchaseReturns = PurchaseReturn::select(
                'code',
                'supplier_code',
                'grn_code',
                'total_amount',
                'remarks',
                'paid_status',
                'return_type',
                'return_date_time'
            )
                ->with([
                    'supplier' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'grn' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'grn.stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                //->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->get();
            return response()->json(['status' => 200, 'purchaseReturns' => $purchaseReturns]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * Author: Suhail Jamaldeen
     * Date: 01.10.2023
     * Logic: Get the salesreturn by Id
     * Summary of show
     * @param mixed $id
     * @throws \Exception
     * @return \Illuminate\Http\JsonResponse|mixed
     */
    public function show($id)
    {
        try {
            $purchaseReturn = PurchaseReturn::select(
                'code',
                'supplier_code',
                'grn_code',
                'total_amount',
                'remarks',
                'paid_status',
                'status',
                'return_type',
                'status',
                'stock_location_code',
                'return_date_time'
            )
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'supplier' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'grn' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'grn.stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'grn.stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('code', $id)
                ->where('is_active', '=', 1)
                ->first();

            return response()->json([
                'status' => 200,
                'purchaseReturn' => $purchaseReturn
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function getAll(Request $request)
    {
        try {
            $purchaseReturn = PurchaseReturn::select(
                'code',
                'supplier_code',
                'grn_code',
                'total_amount',
                'remarks',
                'paid_status',
                'status',
                'return_type',
                'status',
                'stock_location_code',
                'return_date_time'
            )
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'supplier' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'grn' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'grn.stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'grn.stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->where('is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->get();

            return response()->json([
                'status' => 200,
                'purchasesReturn' => $purchaseReturn
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function create()
    {
        //
    }


    /**
     * Author: Suhail Jamaldeen
     * Date: 01.10.2023
     * Logic: Storing Purchase return.
     * Summary of store
     * @param \Illuminate\Http\Request $request
     * @throws \Exception
     * @return \Illuminate\Http\JsonResponse|mixed
     */
    public function store(Request $request)
    {

        try {
            $stock_history = $request->stockHistories;
            if (count($stock_history) > 0) {
                DB::beginTransaction();

                $noteCode = TransactionCode::PURCHASE_RETURN;
                $stockLocationCode = getCurrentLocationCode($request);
                $locationPrefix = substr("$stockLocationCode", -2);
                $max_code = PurchaseReturn::select('code')
                    ->where('stock_location_code', '=', $stockLocationCode)
                    ->max('code');

                $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

                $purchaseReturn = PurchaseReturn::create([
                    'code' => $noteCode . $locationPrefix . $max_id,
                    'supplier_code' => $request->supplier['code'] ?? null,
                    'grn_code' => $request['grn'],
                    'total_amount' => $request->totalAmount,
                    'remarks' => $request->remarks,
                    'stock_location_code' => $stockLocationCode,
                    'paid_status' => 1,
                    'return_type' => $request['returnType'],
                    'return_date_time' => getDateTimeNow(),
                    'created_by' => getUserCode(),
                    'created_at' => getDateTimeNow(),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

                if ($purchaseReturn->return_type == ReturnType::PURCHASE_RETURN_WITH_BILL) {

                    if (config('setting.is_batch_enabled')) {
                        foreach ($stock_history as $stockHistory) {
                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                            $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $purchaseReturn->code, StockHistorySource::PURCHASE_RETURN, $stock);
                        }

                    } else {
                        foreach ($stock_history as $stockHistory) {
                            $stock = Stock::where(
                                'item_code',
                                '=',
                                $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                                )
                                ->where(
                                    'stock_location_code',
                                    '=',
                                    $stockLocationCode
                                )
                                ->where('is_freeze', '=', 0)
                                ->first();
                            $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $purchaseReturn->code, StockHistorySource::PURCHASE_RETURN, $stock);
                        }

                    }

                } else if ($purchaseReturn->return_type == ReturnType::PURCHASE_RETURN_WITHOUT_BILL) {
                    foreach ($stock_history as $stockHistory) {
                        if (config('setting.is_batch_enabled')) {

                            $stock = $this->stock_function->createStock($stockHistory, $purchaseReturn->code, $stockLocationCode);
                            $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $purchaseReturn->code, StockHistorySource::PURCHASE_RETURN, $stock);
                        } else {

                            //$stock = $this->stock_function->createStock($stockHistory, $purchaseReturn->code, $stockLocationCode);

                            $stock = Stock::where(
                                'item_code',
                                '=',
                                $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                            )
                            ->where(
                                'stock_location_code',
                                '=',
                                $stockLocationCode
                            )
                                ->where('is_freeze', '=', 0)
                                ->first();

                                $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                                //$remainingQuantity = $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $purchaseReturn->code, StockHistorySource::PURCHASE_RETURN, $stock);
                        }
                    }
                }

                $supplier = Supplier::where('code', $purchaseReturn->supplier_code)->first();
                $supplier->update([
                    'return_amount' => $supplier['return_amount'] + ($purchaseReturn->total_amount),
                    //'debit_amount' => $supplier['debit_amount'] - ($purchaseReturn->total_amount),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);



                DB::commit();

                return response()->json(
                    [
                        'status' => 200,
                        'PurchaseReturn' => "PurchaseReturn created",
                        'code'=> $purchaseReturn ->code
                    ]
                );
            }
            else {
                return response()->json(
                    [
                        'status' => 200,
                        'PurchaseReturn' => "No Items Selected"
                    ]
                );
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    /**
     * reversal
     * Author: Fathima Sajana
     * Verion: 01
     * Date: 23.10.2023
     * Logic: Reverse the Purchase Return
     * @param  mixed $request
     * @return void
     */
    public function reversal(Request $request)
    {
        try {
            $purchaseReturnCode = $request->transactionCode;
            DB::beginTransaction();

            $purchaseReturn = PurchaseReturn::where('code', '=', $purchaseReturnCode)
                ->where('is_active', '=', 1)
                ->first();

            if (!$purchaseReturn) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Purchase Return Code - ' . $purchaseReturnCode . ' Not Found'
                ]);
            }


            $stockHistories = StockHistory::where('source_code', '=', $purchaseReturnCode)
                ->where('is_active', '=', 1)
                ->get();

            if ($stockHistories->isNotEmpty()) {

                if (config('setting.is_batch_enabled')) {

                    foreach ($stockHistories as $stockHistory) {

                        $stock = $this->stock_function->getExistingStock($stockHistory->stock_code);
                        if ($stock) {
                            $remainingQuantity = $stock->remaining_quantity + $stockHistory->quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                        }

                        $stockHistory->update([
                            'status' => MultiPurposeStatus::REVERSED,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                    }

                } else {

                    foreach ($stockHistories as $stockHistory) {
                        $itemCode = $stockHistory->itemCode ?: $stockHistory->itemMaster->code;
                        $stock = Stock::where('item_code', '=', $itemCode)->first();

                        if ($stock) {
                            $remainingQuantity = $stock->remaining_quantity + $stockHistory->quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                        }
                        $stockHistory->update([
                            'status' => MultiPurposeStatus::REVERSED,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                    }
                }


            }

            if ($purchaseReturn->paid_status !=  PaymentStatus::PAYMENT_GIVING_TO_CUSTOMER) {
                // Reverse the supplier return amount
                $supplier = Supplier::where('code', $purchaseReturn->supplier_code)->first();
                $supplier->update([
                    'return_amount' => $supplier->return_amount - $purchaseReturn->total_amount,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow()
                ]);
            }

            // Change the status of the sales return to reverse
            $purchaseReturn->update([
                'status' => MultiPurposeStatus::REVERSED,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);

            DB::commit();

            return response()->json([
                'status' => 200,
                'message' => $purchaseReturn
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 500,
                'message' => 'An error occurred: ' . $e->getMessage()
            ]);
        }
    }



    public function edit($id)
    {
        //
    }


    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $purchaseReturn = PurchaseReturn::where('is_active', '=', 1)->where('code', $id)->first();
            $supplier = Supplier::where('code', $purchaseReturn->supplier_code)->first();
            $supplier->update([
                'return_amount' => $supplier['return_amount'] - ($purchaseReturn->total_amount),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);

            if ($purchaseReturn) {
                $purchaseReturn->update([
                    'supplier_code' => $request->supplier['code'],
                    // 'grn_code' => $returnValue['grnCode']['code'],
                    'total_amount' => $request->totalAmount,
                    'remarks' => $request->remarks,
                    'return_type' => $request['returnType'],
                    'paid_status' => $request->paidStatus,
                    'return_date_time' => getDateTimeNow(),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow()
                ]);
                foreach ($purchaseReturn->stockHistory as $stockHistory) {
                    if ($purchaseReturn) {

                        $stock_history = $request->stockHistories;
                        if (count($stock_history) > 0) {
                            foreach ($stock_history as $stockHistory) {
                                $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                                $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                                $stock->update([
                                    'remaining_quantity' => $remainingQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow()
                                ]);
                                $this->stock_history_function->updateStockHistory($stockHistory, $purchaseReturn->code, StockHistorySource::PURCHASE_RETURN, $stock);
                            }
                        }

                        $supplier = Supplier::where('code', $purchaseReturn->supplier_code)->first();
                        $supplier->update([
                            'return_amount' => $supplier['return_amount'] > 0 ? $supplier['return_amount'] + ($purchaseReturn->total_amount) : 0 + ($purchaseReturn->total_amount),
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow(),
                        ]);
                    }
                }
            }


            DB::commit();


            return response()->json(['status' => 200, 'PurchaseReturn' => "PurchaseReturn updated"]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
