<?php

namespace App\Http\Controllers\Stock_Controllers;

use App\Enums\LockedStatus;
use App\Enums\StockHistorySource;
use App\Enums\TransactionCode;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Function_Controllers\StockFunction;
use App\Http\Controllers\Function_Controllers\StockHistoryFunction;
use App\Models\Item_Models\ItemMaster;
use App\Models\Stock_Models\MatureStock;
use App\Models\Stock_Models\Stock;
use App\Models\Stock_Models\StockHistory;
use App\Models\Stock_Models\StockIn;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockInController extends Controller
{
    //
    private $stock_history_function;
    private $stock_function;
    public function __construct()
    {
        $this->stock_history_function = new StockHistoryFunction();
        $this->stock_function = new StockFunction();
    }

    public function index(Request $request)
    {

        try {
            $stockIn = StockIn::select(
                'code',
                'stock_location_code',
                'reference_no',
                'reference_name',
                'stock_in_type',
                'batch_no',
                'total_amount',
                'remarks',
                'date',
                'is_locked'
            )
                ->where('stock_ins.is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('created_at', 'DESC')
                ->get();
            return response()->Json([
                'status' => 200,
                'stockIns' => $stockIn
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function show($id)
    {
        try {
            $stockIn = StockIn::select(
                'code',
                'stock_location_code',
                'reference_no',
                'reference_name',
                'stock_in_type',
                'batch_no',
                'sub_total_amount',
                'discount_amount',
                'discount_percentage',
                'stock_in_discount',
                'total_amount',
                'remarks',
                'date',
                'is_locked'
            )
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },

                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stocks' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->where('code', $id)
                ->first();
            return response()->json(
                [
                    'status' => 200,
                    'stockIn' => $stockIn
                ]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function itemPriceSearchforStockIn(Request $request)
    {
        $key = $request->key;
        try {
            $stock = Stock::select(
                'code',
                'code as stockCode',
                'remaining_quantity',
                'wholesale_price',
                'retail_price',
                'cost_price',
                'stock_location_code',
                'item_code'
            )
                ->where([['remaining_quantity', '=', 0]]) //Comment if you don't want to display 0 remaining quantity
                ->where('stock_location_code', '=', $request->header('LocationCode'))
                ->with([
                    'itemMaster' => function ($query) {
                        $query->select(
                            'code',
                            'name',
                            'barcode',
                            'description',
                            'unit_code',
                            'is_service'
                        )
                            ->with([
                                'stocks' => function ($query) {
                                    $query->select(
                                        'code',
                                        'item_code',
                                        'code as stockCode',
                                        'remaining_quantity',
                                        'wholesale_price',
                                        'retail_price',
                                        'cost_price',
                                        'stock_location_code',
                                    );
                                }
                                ,
                                'itemUnit' => function ($query) {
                                    $query->select(
                                        'code',
                                        'name'
                                    )->where('is_active', '=', 1);
                                }
                            ])
                            ->where('is_active', '=', 1);
                    }
                ])
                ->whereHas('itemMaster', function ($query) use ($key) {
                    $query->where('name', 'LIKE', '%' . $key . '%')
                        ->orwhere('barcode', 'LIKE', '%' . $key . '%');

                })
                // ->orwhereHas('itemMaster', function ($query) use ($key) {
                //     $query->where('barcode', 'LIKE', '%' . $key . '%');
                // })

                ->get();

            return response()->json([
                'status' => 200,
                'stock' => $stock

            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    public function stockInLocked(Request $request)
    {
        try {
            $code = $request->code;
            $StockIn = StockIn::where('code', '=', $code)
                ->where('is_locked', '=', 0)
                ->where('is_active', '=', 1)
                ->first();

            $StockIn->update([
                'is_locked' => LockedStatus::LOCKED,
            ]);

            return response()->json(
                [
                    'status' => 200,
                    'stockIn' => $StockIn,
                    'message' => 'StockIn has locked',
                ]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * Auhtor: Suhail Jamaldeen
     * Date: 01.10.2023
     * Logic: Store the Stock In. 
     * @param \Illuminate\Http\Request $request
     * @throws \Exception
     * @return \Illuminate\Http\JsonResponse|mixed
     */
    public function store(Request $request)
    {
        try {

            $stock_history = $request->stockHistories;
            if (count($stock_history) > 0) {

                DB::beginTransaction();

                $stockIn_code = TransactionCode::STOCK_IN;
                $stockLocationCode = getCurrentLocationCode($request);
                $locationPrefix = substr("$stockLocationCode", -2);
                $max_code = StockIn::select('code')
                    ->where('stock_location_code', '=', $stockLocationCode)
                    ->max('code');

                $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

                $stockIn = StockIn::create([
                    'code' => $stockIn_code . $locationPrefix . $max_id,
                    'stock_location_code' => getCurrentLocationCode($request),
                    'reference_no' => $request->referenceNo,
                    'reference_name' => $request->referenceName,
                    'stock_in_type' => $request->stockInType['code'],
                    'batch_no' => $request->batchNo,
                    'date' => $request->date,
                    'sub_total_amount' => $request->subTotalAmount,
                    'discount_amount' => $request->discountAmount ? $request->discountAmount : null,
                    'discount_percentage' => $request->discountPercentage ? $request->discountPercentage : null,
                    'stock_in_discount' => $request->stockInDiscount,
                    'total_amount' => $request->totalAmount,
                    'remarks' => $request->remarks,
                    'created_by' => getUserCode(),
                    'created_at' => getDateTimeNow(),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

                if ($stockIn) {
                    if (config('setting.is_batch_enabled')) {

                        foreach ($stock_history as $stockHistory) {

                            $stock = $this->stock_function->createStock($stockHistory, $stockIn->code, $stockLocationCode);
                            Stock::where('item_code', '=', $stockHistory['itemCode'])
                                ->where('remaining_quantity', '<=', 0)
                                ->where('is_freeze', '=', 0)
                                ->each(function ($oldRecord) {
                                    $matureStock = TransactionCode::MATURE_STOCK;
                                    $maxCode = MatureStock::max('code');
                                    $maxId = $maxCode == null ? config('global.code_value') + 1 : substr("$maxCode", 3) + 1;
                                    $newRecord = $oldRecord->replicate()->fill(
                                        [
                                            'code' => $matureStock . $maxId,
                                            'stock_code' => $oldRecord->code
                                        ]
                                    );
                                    $newRecord->setTable('mature_stocks');
                                    $newRecord->save();
                                    $oldRecord->delete();
                                });

                            $this->stock_history_function->addStockHistory($stockHistory, $stockIn->code, StockHistorySource::STOCK_IN, $stock);
                        }
                    } else {

                        foreach ($stock_history as $stockHistory) {
                            $stock = $this->stock_function->updateOrCreateStock($stockHistory, $stockIn->code, $stockLocationCode);
                            $this->stock_history_function->addStockHistory($stockHistory, $stockIn->code, StockHistorySource::STOCK_IN, $stock);
                        }
                    }
                    if ($request->updatePrice) {

                        $item = ItemMaster::where('code', $stockHistory['itemMaster']['code'])->first();
                        $item->update([
                            'wholesale_price' => $stockHistory['wholesalePrice'] ?? $stockHistory['itemMaster']['wholesalePrice'],
                            'retail_price' => $stockHistory['retailPrice'] ?? $stockHistory['itemMaster']['retailPrice'],
                            'dealer_price' => $stockHistory['dealerPrice'] ?? $stockHistory['itemMaster']['dealerPrice'],
                            'least_price' => $stockHistory['leastPrice'] ?? $stockHistory['itemMaster']['leastPrice'],
                            'cost_price' => $stockHistory['costPrice'] ?? $stockHistory['itemMaster']['costPrice'],
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow(),
                        ]);
                    }


                }
                DB::commit();
                return response()->json([
                    'status' => 200,
                    'message' => "Stock In Created"
                ], 200);
            } else {
                return response()->json(
                    [
                        'status' => 200,
                        'Stock In' => "No Items Selected"
                    ]
                );
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }


    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $stockLocation = getCurrentLocationCode($request);

            $stockIn = StockIn::where('code', $id)->first();
            $stockIn->update([
                'reference_no' => $request->referenceNo,
                'reference_name' => $request->referenceName,
                'stock_in_type' => $request->stockInType,
                'batch_no' => $request->batchNo,
                'date' => $request->date,
                'sub_total_amount' => $request->subTotalAmount,
                'discount_amount' => $request->discountAmount,
                'discount_percentage' => $request->discountPercentage,
                'stock_in_discount' => $request->stockInDiscount,
                'total_amount' => $request->totalAmount,
                'remarks' => $request->remarks,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            if ($stockIn) {

                $stockHistories = $request->stockHistories;

                if (count($stockHistories) > 0) {
                    foreach ($stockHistories as $stockHistory) {
                        if (isset($stockHistory['code'])) { //Existing Item  in stock History for Edit

                            //Editing Existing stock History Details
                            $stockHistoryExisting = StockHistory::where('code', $stockHistory['code'])
                                ->first();

                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                            //Quantity Calculation for Stock
                            $stockHistoryQuantity = $stockHistoryExisting->quantity;
                            $stockCurrentQuantity = $stock->remaining_quantity;
                            $newQuantity = $stockHistory['quantity'];
                            $soldQuantity = ($stockHistoryQuantity - $stockCurrentQuantity);
                            $updatingQuanity = $newQuantity - $soldQuantity;

                            $stock->update([
                                'wholesale_price' => $stockHistory['wholesalePrice'] ?? null,
                                'retail_price' => $stockHistory['retailPrice'] ?? null,
                                'dealer_price' => $stockHistory['dealerPrice'] ?? null,
                                'least_price' => $stockHistory['leastPrice'] ?? null,
                                'cost_price' => $stockHistory['costPrice'] ?? null,
                                'discount_price' => $stockHistory['discountPrice'] ?? null,
                                'discount' => $stockHistory['discount'] ?? null,
                                'remaining_quantity' => $updatingQuanity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow(),
                            ]);
                            StockHistory::where('code', '=', $stockHistory['code'])
                                ->update([
                                    'remaining_quantity' => $stock['remaining_quantity'] ?? null,
                                    'wholesale_price' => $stockHistory['wholesalePrice'] ?? null,
                                    'retail_price' => $stockHistory['retailPrice'] ?? null,
                                    'dealer_price' => $stockHistory['dealerPrice'] ?? null,
                                    'least_price' => $stockHistory['leastPrice'] ?? null,
                                    'cost_price' => $stockHistory['costPrice'] ?? null,
                                    'quantity' => $stockHistory['quantity'] ?? null,
                                    'unit_sold_price' => $stockHistory['unitSoldPrice'] ?? null,
                                    'discount_amount' => $stockHistory['discountAmount'] ?? null,
                                    'discount_percentage' => $stockHistory['discountPercentage'] ?? null,
                                    'discount_total' => $stockHistory['discountTotal'] ?? null,
                                    'total_amount' => $stockHistory['totalAmount'] ?? null,
                                    'sub_total_amount' => $stockHistory['subTotalAmount'] ?? null,
                                    //$stockHistory['quantity'] * $stockHistory['itemMaster']['costPrice'],
                                    'warranty_due_date' => $stockHistory['warrantyDueDate'] ?? null,
                                    'warranty_periods' => $stockHistory['warrantyPeriods'] ?? null,
                                    'expiry_date' => $stockHistory['expiryDate'] ?? null,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow(),
                                ]);
                        } else {
                            $stock = $this->stock_function->createStock($stockHistory, $stockIn->code, $stockLocation);
                            $adddedStockHistory = $this->stock_history_function->addStockHistory($stockHistory, $stockIn->code, StockHistorySource::STOCK_IN, $stock);
                        }
                        //Move zero stocks to Mature Stocks Table
                        Stock::where('item_code', '=', $stockHistory['itemCode'])
                            ->where('remaining_quantity', '<=', 0)
                            ->each(function ($oldRecord) {
                                $matureStock = TransactionCode::MATURE_STOCK;
                                $maxCode = MatureStock::max('code');
                                $maxId = $maxCode == null ? config('global.code_value') + 1 : substr("$maxCode", 3) + 1;
                                $newRecord = $oldRecord->replicate()->fill(
                                    [
                                        'code' => $matureStock . $maxId,
                                        'stock_code' => $oldRecord->code
                                    ]
                                );
                                $newRecord->setTable('mature_stocks');
                                $newRecord->save();
                                $oldRecord->delete();
                            });
                    }
                }

            }

            DB::commit();
            return response()->json([
                'status' => 200,
                'message' => "GRN Updated"
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }
}