<?php

namespace App\Http\Controllers\Stock_Controllers;

use App\Enums\DefaultValues;
use App\Enums\FreezeStatus;
use App\Enums\LockedStatus;
use App\Enums\PaymentStatus;
use App\Enums\SourceCode;
use App\Enums\StockHistorySource;
use App\Http\Controllers\Function_Controllers\StockFunction;
use App\Models\Stock_Models\MatureStock;
use Illuminate\Http\Request;
use App\Enums\TransactionCode;
use App\Models\Stock_Models\Grn;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Function_Controllers\StockHistoryFunction;
use App\Models\Stock_Models\StockHistory;
use App\Models\Item_Models\ItemMaster;
use App\Models\Stock_Models\Stock;
use App\Models\Supplier_Models\Supplier;
use Exception;
use League\CommonMark\Extension\CommonMark\Node\Block\ThematicBreak;

class GrnController extends Controller
{
    private $stock_history_function;
    private $stock_function;
    public function __construct()
    {
        $this->stock_history_function = new StockHistoryFunction();
        $this->stock_function = new StockFunction();
    }


    public function index(Request $request)
    {

        try {
            $grn = Grn::select(
                'grns.code',
                'supplier_code',
                'stock_location_code',
                'total_amount',
                'invoice_code',
                'invoice_date',
                'grn_date',
                'is_freeze',
                'remarks',
                'is_locked'
            )
                ->with('supplier')
                ->where('grns.is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('grns.code', 'DESC')
                ->get();
            return response()->Json(['status' => 200, 'grn' => $grn]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * show
     *
     * @param  mixed $id
     * @return void
     * Author: Suhail Jamaldeen
     * Date: 13.12.2022
     * Version: 01
     * Logic: Get the drn details based on the GRN id
     */
    public function show($id)
    {
        try {
            $grn = Grn::select(
                'code',
                'supplier_code',
                'invoice_code',
                'invoice_date',
                'grn_date',
                'sub_total_amount',
                'discount_amount',
                'discount_percentage',
                'grn_discount',
                'total_amount',
                'paid_amount',
                'is_freeze',
                'remarks',
                'stock_location_code',
                'is_locked'
            )
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'supplier.supplierAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stocks' => function ($query) {
                        $query
                        ->where('is_freeze', '=', 0)
                        ->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'purchaseReturns' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'paymentVouchers' => function ($query) {
                        // $query->where('is_active', '=', 1);
                    }
                ])

                ->where('is_active', '=', 1)
                ->where('code', $id)
                ->first();
            return response()->json(
                [
                    'status' => 200,
                    'grn' => $grn
                ]
            );
        } catch (Exception $e) {
            throw new Exception($e);
        }
    }


    public function grnSearch($key)
    {

        try {
            $grn = Grn::select('grns.code', 'supplier_code', 'suppliers.name', 'total_amount', 'invoice_code', 'invoice_date', 'grn_date')
                ->join('suppliers', 'suppliers.code', '=', 'grns.supplier_code')
                ->where([['grns.is_active', '=', 1]])
                ->where(function ($query) use ($key) {
                    $query->Where('grns.code', 'like', "%$key%")
                        ->orWhere('supplier_code', 'like', "%$key%")
                        ->orWhere('suppliers.name', 'like', "%$key%")
                        ->orWhere('invoice_code', 'like', "%$key%");
                })->get();
            return response()->Json(['status' => 200, 'grn' => $grn]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    public function create()
    {
        //
    }


    // public function store(Request $request)
    // {
    //     DB::beginTransaction();
    //     try {
    //         $grn_code = TransactionCode::GRN;
    //         $max_code = Grn::max('code');
    //         $max_id = $max_code == null ? config('global.code_value') + 1 : substr("$max_code", 3) + 1;

    //         $grn = Grn::create([
    //             'code' =>  $grn_code . $max_id,
    //             'supplier_code' => $request->supplier['code'],
    //             'invoice_code' => $request->invoiceCode,
    //             'invoice_date' => $request->invoiceDate,
    //             'grn_date' => $request->grnDate,
    //             'total_amount' => $request->totalAmount,
    //             'discount' => $request->discount,
    //             'sub_total_amount' => $request->subTotalAmount,
    //             'created_by' => getUserCode(),
    //             'created_at' => getDateTimeNow(),
    //             'updated_by' => getUserCode(),
    //             'updated_at' => getDateTimeNow(),
    //         ]);

    //         if ($grn) {

    //             $stock_history = $request->stockHistory;
    //             if (count($stock_history) > 0) {
    //                 foreach ($stock_history as $stockHistory) {
    //                     $this->stock_history_function->addStockHistory($stockHistory, $grn->code, 'GRN');
    //                 }
    //             }
    //             if ($request->updatePrice) {

    //                 $item = ItemMaster::where('code',  $stockHistory['itemMaster']['code'])->first();
    //                 $item->update([
    //                     'wholesale_price' => $stockHistory['wholesalePrice'] ?? $stockHistory['itemMaster']['wholesalePrice'],
    //                     'retail_price' => $stockHistory['retailPrice'] ?? $stockHistory['itemMaster']['retailPrice'],
    //                     'dealer_price' => $stockHistory['dealerPrice'] ?? $stockHistory['itemMaster']['dealerPrice'],
    //                     'least_price' => $stockHistory['leastPrice'] ?? $stockHistory['itemMaster']['leastPrice'],
    //                     'cost_price' => $stockHistory['costPrice'] ?? $stockHistory['itemMaster']['costPrice'],
    //                     'updated_at' => config('global.current_time'),
    //                 ]);
    //             }
    //         }


    //         DB::commit();
    //         return response()->json([
    //             'status' => 200,
    //             'message' => "GRN Created"
    //         ], 200);
    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         throw new Exception($e);
    //     }
    // }

    public function grnLocked(Request $request)
    {
        try {
            $code = $request->code;
            $grn = Grn::where('code', '=', $code)
                ->where('is_locked', '=', 0)
                ->where('is_active', '=', 1)
                ->first();

            $grn->update([
                'is_locked' => LockedStatus::LOCKED,
            ]);

            return response()->json(
                [
                    'status' => 200,
                    'grn' => $grn,
                    'message' => 'Grn has locked',
                ]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * store
     * Author: Suhail Jamaldeen
     * Date: 18.12.2022
     * Version: 01
     * Logic: Storing the GRN and create record in stock and remove the lines from stock
     * if the remaning quantity is zero
     * @param  mixed $request
     * @return void
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $stock_history = $request->stockHistories;
            if (count($stock_history) > 0) {
                $grn_code = TransactionCode::GRN;
                $stockLocationCode = getCurrentLocationCode($request);
                $locationPrefix = substr("$stockLocationCode", -2);
                $max_code = Grn::select('code')
                    ->where('stock_location_code', '=', $stockLocationCode)
                    ->max('code');

                $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

                $grn = Grn::create([
                    'code' => $grn_code . $locationPrefix . $max_id,
                    'supplier_code' => $request->supplier['code'],
                    'stock_location_code' => getCurrentLocationCode($request),
                    'invoice_code' => $request->invoiceCode,
                    'invoice_date' => $request->invoiceDate,
                    'grn_date' => $request->grnDate,
                    'grn_discount' => $request->grnDiscount,
                    'discount_amount' => $request->discountAmount,
                    'discount_percentage' => $request->discountPercentage,
                    'total_amount' => $request->totalAmount,
                    'paid_amount' => $request->paidAmount,
                    'remarks' => $request->remarks,
                    'paid_status' => PaymentStatus::PAYMENT_NOT_DONE,
                    'created_by' => getUserCode(),
                    'created_at' => getDateTimeNow(),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

                if ($grn) {

                    // If the client request not to show batch in price. We can use this.
                    if (config('setting.is_batch_enabled')) {

                        foreach ($stock_history as $stockHistory) {

                            $stock = $this->stock_function->createStock($stockHistory, $grn->code, $stockLocationCode);
                            Stock::where('item_code', '=', $stockHistory['itemCode'])
                                ->where('remaining_quantity', '<=', 0)
                                ->each(function ($oldRecord) {
                                    $matureStock = TransactionCode::MATURE_STOCK;
                                    $maxCode = MatureStock::max('code');
                                    $maxId = $maxCode == null ? config('global.code_value') + 1 : substr("$maxCode", 3) + 1;
                                    $newRecord = $oldRecord->replicate()->fill(
                                        [
                                            'code' => $matureStock . $maxId,
                                            'stock_code' => $oldRecord->code
                                        ]
                                    );
                                    $newRecord->setTable('mature_stocks');
                                    $newRecord->save();
                                    $oldRecord->delete();
                                });

                            $this->stock_history_function->addStockHistory($stockHistory, $grn->code, StockHistorySource::GRN, $stock);

                            if ($request->updatePrice) {

                                $item = ItemMaster::where('code', $stockHistory['itemMaster']['code'])->first();
                                $item->update([
                                    'wholesale_price' => $stockHistory['wholesalePrice'] ?? $stockHistory['itemMaster']['wholesalePrice'],
                                    'retail_price' => $stockHistory['retailPrice'] ?? $stockHistory['itemMaster']['retailPrice'],
                                    'dealer_price' => $stockHistory['dealerPrice'] ?? $stockHistory['itemMaster']['dealerPrice'],
                                    'least_price' => $stockHistory['leastPrice'] ?? $stockHistory['itemMaster']['leastPrice'],
                                    'cost_price' => $stockHistory['costPrice'] ?? $stockHistory['itemMaster']['costPrice'],
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow(),
                                ]);
                            }
                        }
                    } else {   //freezing function doing only for non batch stock

                        foreach ($stock_history as $stockHistory) {

                            $stock = $this->stock_function->updateOrCreateStock($stockHistory, $grn->code, $stockLocationCode);
                            $this->stock_history_function->addStockHistory($stockHistory, $grn->code, StockHistorySource::GRN, $stock);

                            if ($request->updatePrice) {

                                $item = ItemMaster::where('code', $stockHistory['itemMaster']['code'])->first();
                                $item->update([
                                    'wholesale_price' => $stockHistory['wholesalePrice'] ?? $stockHistory['itemMaster']['wholesalePrice'],
                                    'retail_price' => $stockHistory['retailPrice'] ?? $stockHistory['itemMaster']['retailPrice'],
                                    'dealer_price' => $stockHistory['dealerPrice'] ?? $stockHistory['itemMaster']['dealerPrice'],
                                    'least_price' => $stockHistory['leastPrice'] ?? $stockHistory['itemMaster']['leastPrice'],
                                    'cost_price' => $stockHistory['costPrice'] ?? $stockHistory['itemMaster']['costPrice'],
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow(),
                                ]);
                            }
                        }
                    }

                    $supplier = Supplier::where('code', $grn->supplier_code)->first();
                    $supplier->update([
                        'grn_total' => $grn['total_amount'] + $supplier['grn_total'],
                        //'received_amount' => $grn['paid_amount'] ?: 0,
                        'debit_amount' => $grn['paid_amount'] == 0 ? ($supplier['debit_amount'] + $grn['total_amount']) : $supplier['debit_amount'],
                        //'debit_amount' => $supplier['debit_amount'] + ($grn->total_amount - $grn-> paid_amount),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow(),
                    ]);
                }


                DB::commit();
                return response()->json([
                    'status' => 200,
                    'message' => "GRN Created"
                ], 200);
            }
        } catch (Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function edit($id)
    {
        //
    }


    /**
     * update
     * Author: Suhail Jamaldeen
     * Date: 18.12.2022
     * Version: 01
     * Logic: Update Grn
     * @param  mixed $request
     * @param  mixed $id
     * @return void
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        // try {
        $grn = Grn::where('code', $id)
            ->first();
        $stockLocation = getCurrentLocationCode($request);
        if ($grn) {
            $supplier = Supplier::where('code', $grn->supplier_code)->first();

            $supplier->update([
                'grn_total' => (($supplier['grn_total'] - $grn['total_amount'])),

                // when newly start this function for changing supplier (already existing grn supplier) , call this one for reducing debit_amount from that supplier
                'debit_amount' => $supplier['debit_amount'] - (($grn->total_amount) - ($grn->paid_amount)),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
        }
        $grn = Grn::where('code', $id)->first();
        $grn->update([
            'supplier_code' => $request->supplier['code'],
            'invoice_code' => $request->invoiceCode,
            'invoice_date' => $request->invoiceDate,
            'grn_date' => $request->grnDate,
            'sub_total_amount' => $request->subTotalAmount,
            'discount_amount' => $request->discountAmount ? $request->discountAmount : null,
            'discount_percentage' => $request->discountPercentage ? $request->discountPercentage : null,
            'grn_discount' => $request->grnDiscount,
            'total_amount' => $request->totalAmount,
            'remarks' => $request->remarks,
            //'paid_amount' => $request->paidAmount,
            'updated_by' => getUserCode(),
            'updated_at' => getDateTimeNow(),
        ]);
        if ($grn) {

            $stock_Histories = $request->stockHistories;

            if (count($stock_Histories) > 0) {

                foreach ($stock_Histories as $stockHistory) {

                    if (isset($stockHistory['code']))
                    {
                        //Existing Item  in stock History for Edit
                        //Editing Existing stock History Details

                        $stockHistoryExisting = StockHistory::where('code', $stockHistory['code'])
                            ->first();

                        $stock = null;

                        if (config('setting.is_batch_enabled'))
                        {
                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                        }

                        else
                        {
                            $stock = Stock::where('item_code', '=', $stockHistory['itemCode'])
                            ->where('stock_location_code', '=', $stockLocation)
                            ->where('code', '=',  $stockHistory['stockCode'])
                            ->first();
                        }
                            if ($stock != null) {
                            //Quantity Calculation for Stock
                            $stockHistoryQuantity = $stockHistoryExisting->quantity;
                            $stockCurrentQuantity = $stock->remaining_quantity;
                            $newQuantity = $stockHistory['quantity'];
                            $soldQuantity = ($stockHistoryQuantity - $stockCurrentQuantity);
                            $updatingQuanity = $newQuantity - $soldQuantity;

                            $stock->update([
                                'wholesale_price' => $stockHistory['wholesalePrice'] ?? null,
                                'retail_price' => $stockHistory['retailPrice'] ?? null,
                                'dealer_price' => $stockHistory['dealerPrice'] ?? null,
                                'least_price' => $stockHistory['leastPrice'] ?? null,
                                'cost_price' => $stockHistory['costPrice'] ?? null,
                                'discount_price' => $stockHistory['discountPrice'] ?? null,
                                'discount' => $stockHistory['discount'] ?? null,
                                'remaining_quantity' => $updatingQuanity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow(),
                            ]);
                            StockHistory::where('code', '=', $stockHistory['code'])
                                ->update([
                                    'remaining_quantity' => $stock['remaining_quantity'] ?? null,
                                    'wholesale_price' => $stockHistory['wholesalePrice'] ?? null,
                                    'retail_price' => $stockHistory['retailPrice'] ?? null,
                                    'dealer_price' => $stockHistory['dealerPrice'] ?? null,
                                    'least_price' => $stockHistory['leastPrice'] ?? null,
                                    'cost_price' => $stockHistory['costPrice'] ?? null,
                                    'quantity' => $stockHistory['quantity'] ?? null,
                                    'unit_sold_price' => $stockHistory['unitSoldPrice'] ?? null,
                                    'discount_amount' => $stockHistory['discountAmount'] ?? null,
                                    'discount_percentage' => $stockHistory['discountPercentage'] ?? null,
                                    'discount_total' => $stockHistory['discountTotal'] ?? null,
                                    'total_amount' => $stockHistory['totalAmount'] ?? null,
                                    'sub_total_amount' => $stockHistory['subTotalAmount'] ?? null,
                                    //$stockHistory['quantity'] * $stockHistory['itemMaster']['costPrice'],
                                    'warranty_due_date' => $stockHistory['warrantyDueDate'] ?? null,
                                    'warranty_periods' => $stockHistory['warrantyPeriods'] ?? null,
                                    'expiry_date' => $stockHistory['expiryDate'] ?? null,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow(),
                                ]);
                        }
                    }
                        else
                    {
                        // If the item is already available in stock(come from when inserting item)but this item don't have any stockHistory(it means that item has no grn)
                        //this item doesn't has stockHistory
                        $stock = Stock::where('item_code', '=', $stockHistory['itemCode'])
                            ->where('stock_location_code', '=', $stockLocation)
                            ->where('is_freeze', '=', FreezeStatus::NON_FREEZED)
                            ->first();

                            if ($stock) {
                                $stockCurrentQuantity = $stock->remaining_quantity;
                                $quantity = $stockHistory['quantity'];
                                $newQuantity = ($stockCurrentQuantity + $quantity);

                                $stock->update([
                                    'wholesale_price' => $stockHistory['wholesalePrice'] ?? null,
                                    'retail_price' => $stockHistory['retailPrice'] ?? null,
                                    'dealer_price' => $stockHistory['dealerPrice'] ?? null,
                                    'least_price' => $stockHistory['leastPrice'] ?? null,
                                    'cost_price' => $stockHistory['costPrice'] ?? null,
                                    'discount_price' => $stockHistory['discountPrice'] ?? null,
                                    'discount' => $stockHistory['discount'] ?? null,
                                    'remaining_quantity' => $newQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow(),
                                ]);
                            }
                            else {
                                $stock = $this->stock_function->createStock($stockHistory, $grn->code, $stockLocation);
                            }

                        $adddedStockHistory = $this->stock_history_function->addStockHistory($stockHistory, $grn->code, StockHistorySource::GRN, $stock);
                    }
                    // if (config('setting.is_batch_enabled')) {
                    //     //Move zero stocks to Mature Stocks Table omly if the stock is batch enabled
                    //     Stock::where('item_code', '=', $stockHistory['itemCode'])
                    //         ->where('remaining_quantity', '<=', 0)
                    //         ->each(function ($oldRecord) {
                    //             $matureStock = TransactionCode::MATURE_STOCK;
                    //             $maxCode = MatureStock::max('code');
                    //             $maxId = $maxCode == null ? config('global.code_value') + 1 : substr("$maxCode", 3) + 1;
                    //             $newRecord = $oldRecord->replicate()->fill(
                    //                 [
                    //                     'code' => $matureStock . $maxId,
                    //                     'stock_code' => $oldRecord->code
                    //                 ]
                    //             );
                    //             $newRecord->setTable('mature_stocks');
                    //             $newRecord->save();
                    //             $oldRecord->delete();
                    //         });
                    // }
                }
            }

            if ($request->updatePrice) {

                $item = ItemMaster::where('code', $stockHistory['itemMaster']['code'])->first();
                $item->update([
                    'wholesale_price' => $stockHistory['wholesalePrice'] ?? $stockHistory['itemMaster']['wholesalePrice'],
                    'retail_price' => $stockHistory['retailPrice'] ?? $stockHistory['itemMaster']['retailPrice'],
                    'dealer_price' => $stockHistory['dealerPrice'] ?? $stockHistory['itemMaster']['dealerPrice'],
                    'least_price' => $stockHistory['leastPrice'] ?? $stockHistory['itemMaster']['leastPrice'],
                    'cost_price' => $stockHistory['costPrice'] ?? $stockHistory['itemMaster']['costPrice'],
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);
            }

            // calling this debit_amount for newly changed supplier (from existing supplier to updated supplier), this supplier alredy had debit_amount adding new amount through this debit_amount , if it is not call other fuction ($grn->total_amount - $grn->paid_amount)
            $supplier = Supplier::where('code', $grn->supplier_code)->first();
            $supplier->update([
                //'grn_total' => $grn['paid_amount'] == 0 ? ($supplier['debit_amount'] + $grn['total_amount']) : $supplier['debit_amount'],
                //'debit_amount' => $grn['paid_amount'] == 0 ? ($supplier['debit_amount'] + $grn['total_amount']) : $supplier['debit_amount'],
                'grn_total' => ($supplier['grn_total'] + $grn['total_amount']),
                //'received_amount' => $grn['paid_amount'] ? $grn['paid_amount'] : 0,
                'debit_amount' => $supplier['debit_amount'] ? ($supplier['debit_amount'] + ($grn->total_amount - $grn->paid_amount)) : ($supplier['debit_amount'] + ($grn->total_amount - $grn->paid_amount)),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
        }

        DB::commit();
        return response()->json([
            'status' => 200,
            'message' => "GRN Updated"
        ], 200);
        // } catch (\Exception $e) {
        //     DB::rollBack();
        //     throw new Exception($e);
        // }
    }


    public function destroy($id)
    {
        try {
            $grn = Grn::select('is_active')->findOrFail($id);
            $grn->update([
                'is_active' => 0
            ]);

            return response()->json([
                'status' => 200,
                'message' => 'GRN Delete'

            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    /**
     * getNotPaidGrn
     * Author: Suhail Jamaldeen
     * Date: 20.12.2022
     * Version: 01
     * Logic: Get the GRNs which no payment done.
     *
     * @return void
     */
    public function getNotPaidGrn(Request $request)
    {
        try {
            $grn = Grn::select(
                'code',
                'stock_location_code',
                'supplier_code',
                'total_amount',
                'invoice_code',
                'invoice_date',
                'grn_date',
                'is_freeze',
                'is_locked'
            )
                ->with('supplier')
                ->where('is_active', '=', 1)
                ->where('paid_status', '=', PaymentStatus::PAYMENT_NOT_DONE)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('grns.code', 'DESC')
                ->get();
            return response()->Json(
                ['status' => 200, 'grn' => $grn]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * getPaidGrn
     * Author: Suhail Jamaldeen
     * Date: 20.12.2022
     * Version: 01
     * Logic: Get the partial and fully paid Grn
     *
     * @return void
     */
    public function getPaidGrn(Request $request)
    {
        try {
            $grn = Grn::select(
                'code',
                'supplier_code',
                'stock_location_code',
                'total_amount',
                'paid_amount',
                'invoice_code',
                'invoice_date',
                'is_freeze',
                'grn_date',
                'is_locked'
            )
                ->with('supplier')
                ->where('grns.is_active', '=', 1)
                ->where('grns.paid_status', '=', PaymentStatus::PAYMENT_PARTIAL)
                ->orWhere('grns.paid_status', '=', PaymentStatus::PAYMENT_COMPLETED)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('grns.code', 'DESC')
                ->get();
            return response()->Json(
                ['status' => 200, 'grn' => $grn]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }
    public function outstandingGrnForSupplier($key)
    {
        try {
            $grn = Grn::select(
                'grns.code',
                'supplier_code',
                'suppliers.name',
                'total_amount',
                'invoice_code',
                'invoice_date',
                'grn_date',
                'grns.paid_amount'
            )
                ->join('suppliers', 'suppliers.code', '=', 'grns.supplier_code')
                ->where([
                    ['grns.is_active', '=', 1],
                ])
                ->whereColumn(
                    'grns.total_amount',
                    '>',
                    'grns.paid_amount'
                )
                ->where(function ($query) use ($key) {
                    $query->where('grns.code', '=', $key)
                          ->orWhere('supplier_code', '=', $key)
                          ->orWhere('invoice_code', '=', $key);
                })
                ->get();
            return response()->Json(['status' => 200, 'grn' => $grn]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * Author: Suhail Jamaldeen
     * Date: 21.02.2022
     * Version: 01
     * Logic: Get all the GRNFor Supplier. This is used for purchase return
     * getGrnForSupplierPurchaseReturn
     *
     * @param  mixed $key : supplier Code
     * @return void
     */
    function getGrnForSupplierPurchaseReturn($key)
    {
        try {
            $grn = Grn::select(
                'code',
                'supplier_code',
                'invoice_code',
                'invoice_date',
                'grn_date',
                'sub_total_amount',
                'discount_amount',
                'discount_percentage',
                'grn_discount',
                'is_freeze',
                'total_amount',
                'paid_amount'
            )
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'supplier' => function ($query) {
                        $query->select('code', 'name',)->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stocks' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->where('supplier_code', $key)
                ->get();
            return response()->json(
                [
                    'status' => 200,
                    'grn' => $grn
                ]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }
}
