<?php

namespace App\Http\Controllers\StockIssue_Controllers;

use App\Enums\LockedStatus;
use App\Enums\StockHistorySource;
use App\Enums\TransactionCode;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Function_Controllers\StockFunction;
use App\Http\Controllers\Function_Controllers\StockHistoryFunction;
use App\Models\Customer_Models\Customer;
use App\Models\Stock_Models\Stock;
use App\Models\Stock_Models\StockHistory;
use App\Models\StockIssue_Models\StockIssue;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockIssueController extends Controller
{
    private $stock_history_function;
    private $stock_function;

    public function __construct()
    {
        $this->stock_function = new StockFunction();

        $this->stock_history_function = new StockHistoryFunction();
    }
    public function index(Request $request)
    {
        try {

            $stockIssues = StockIssue::select(
                'code',
                'stock_issue_type_code',
                'stock_location_code',
                'reference_code',
                'destination_code',
                'is_locked',
                'remarks',
                'total',
                'created_at'
            )
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockIssueType' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'department' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'customer.route.salesRep' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('created_at', 'DESC')
                ->get();
            return response()->json(['status' => 200, 'stockIssues' => $stockIssues]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function show($id)
    {
        try {

            $stockIssue = StockIssue::select(
                'code',
                'stock_issue_type_code',
                'reference_code',
                'destination_code',
                'remarks',
                'total',
                'stock_location_code',
                'created_at',
                'is_locked'
            )
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockIssueType' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'department' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'customer.route.salesRep' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('code', '=', $id)
                ->where('is_active', '=', 1)
                ->first();

            return response()->json(['status' => 200, 'stockIssue' => $stockIssue]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function create()
    {
        //
    }

    public function stockIssueLocked(Request $request)
    {
        try {
            $code = $request->code;
            $StockIssue = StockIssue::where('code', '=', $code)
                ->where('is_locked', '=', 0)
                ->where('is_active', '=', 1)
                ->first();

            $StockIssue->update([
                'is_locked' => LockedStatus::LOCKED,
            ]);

            return response()->json(
                [
                    'status' => 200,
                    'stockIssue' => $StockIssue,
                    'message' => 'StockIssue has locked',
                ]
            );
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * Author: Suhail Jamaldeen
     * Version: 01
     * Date: 15.10.2023
     * Logic: Store Stock Issue
     * Summary of store
     * @param \Illuminate\Http\Request $request
     * @throws \Exception
     * @return \Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $stock_history = $request->stockHistories;
            DB::beginTransaction();

            if (count($stock_history) > 0) {

                $sales = TransactionCode::STOCK_ISSUE;
                $stockLocationCode = getCurrentLocationCode($request);
                $locationPrefix = substr("$stockLocationCode", -2);
                $max_code = StockIssue::select('code')
                    ->where('stock_location_code', '=', $stockLocationCode)
                    ->max('code');

                $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

                $type = $request->type;
                $stockIssue = StockIssue::create([
                    'code' => $sales . $locationPrefix . $max_id,
                    'reference_code' => $request->referenceCode,
                    'destination_code' => $request->customer['code'],
                    'stock_issue_type_code' => $request->stockIssueType['code'],
                    'remarks' => $request->remarks,
                    'total' => $request->totalAmount,
                    'stock_location_code' => getCurrentLocationCode($request),
                    'is_department' => $type == 'department' ? 1 : 0,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                    'created_by' => getUserCode(),
                    'created_at' => getDateTimeNow()

                ]);
                if ($stockIssue) {
                    if (config('setting.is_batch_enabled')) {

                        foreach ($stock_history as $stockHistory) {

                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                            $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $stockIssue->code, StockHistorySource::STOCK_ISSUE, $stock);
                        }
                    } else {
                        foreach ($stock_history as $stockHistory) {

                            $stock = Stock::where(
                                'item_code',
                                '=',
                                $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                            )->where(
                                    'stock_location_code',
                                    '=',
                                    $stockLocationCode
                                )
                                ->where('is_freeze', '=', 0)
                                ->first();

                            $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $stockIssue->code, StockHistorySource::STOCK_ISSUE, $stock);
                        }
                    }
                    if ($type == 'customer') {
                        $this->customerCreditBalance($stockIssue);

                    }


                }


                DB::commit();
                return response([
                    'status' => 200,
                    'Message' => 'stockIssue Created'
                ]);
            } else {
                return response([
                    'status' => 200,
                    'Message' => 'Stock History Not Available'
                ]);
            }

        } catch (\Exception $e) {
            DB::rollback();
            throw new Exception($e);
        }
    }


    public function customerCreditBalance($stockIssue)
    {
        $customer = Customer::where('code', $stockIssue->destination_code)->first();
        $customer->update([
            'credit_amount' => $customer['credit_amount'] - ($stockIssue->total),
            'updated_by' => getUserCode(),
            'updated_at' => getDateTimeNow(),
        ]);
    }



    public function edit($id)
    {
        //
    }


    /**
     * Author: Suhail Jamaldeen
     * Version: 02
     * Date:16.10.2023
     * Logic: Update Stock Issue.
     * Summary of update
     * @param \Illuminate\Http\Request $request
     * @param mixed $id
     * @throws \Exception
     * @return \Illuminate\Http\JsonResponse|mixed
     */
    public function update(Request $request, $id)
    {
        try {
            $stock_history = $request->stockHistories;

            if (count($stock_history) > 0) {
                DB::beginTransaction();

                $stockIssue = StockIssue::where('code', $id)->first();
                //No need to update stock location
                $stockIssue->update([
                    'reference_code' => $request->referenceCode,
                    'destination_code' => $request->customer['code'],
                    'stock_issue_type_code' => $request->stockIssueType['code'],
                    'remarks' => $request->remarks,
                    'total' => $request->totalAmount,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

                if ($stockIssue) {
                    if (config('setting.is_batch_enabled')) {

                        foreach ($stock_history as $stockHistory) {
                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);

                            if (isset($stockHistory['code'])) { //Existing Item  in stock History for Edit
                                //Editing Existing stock History Details
                                $stockHistoryExisting = StockHistory::where('code', $stockHistory['code'])
                                    ->first();
                                $stockHistoryExistingQuanity = $stockHistoryExisting->quantity;
                                $remainingQuantity = ($stock->remaining_quantity + $stockHistoryExistingQuanity) - $stockHistory['quantity'];
                                $stock->update([
                                    'remaining_quantity' => $remainingQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow()
                                ]);
                                $this->stock_history_function->updateStockHistory($stockHistory, $stockIssue->code, StockHistorySource::STOCK_ISSUE, $stock);

                            } else {
                                $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                                $stock->update([
                                    'remaining_quantity' => $remainingQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow()
                                ]);
                                $this->stock_history_function->addStockHistory($stockHistory, $stockIssue->code, StockHistorySource::STOCK_ISSUE, $stock);
                            }
                        }
                    } else {
                        foreach ($stock_history as $stockHistory) {
                            if (isset($stockHistory['code'])) { //Existing Item  in stock History for Edit

                                //Editing Existing stock History Details
                                $stockHistoryExisting = StockHistory::where('code', $stockHistory['code'])
                                    ->first();
                                $stockHistoryExistingQuanity = $stockHistoryExisting->quantity;


                                $stock = Stock::where(
                                    'item_code',
                                    '=',
                                    $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                                )->where(
                                        'stock_location_code',
                                        '=',
                                        $stockIssue->stock_location_code
                                    )
                                    ->first();
                                $remainingQuantity = ($stock->remaining_quantity + $stockHistoryExistingQuanity) - $stockHistory['quantity'];
                                $stock->update([
                                    'remaining_quantity' => $remainingQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow()
                                ]);
                                $this->stock_history_function->updateStockHistory($stockHistory, $stockIssue->code, StockHistorySource::STOCK_ISSUE, $stock);
                            } else {
                                $stock = Stock::where(
                                    'item_code',
                                    '=',
                                    $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                                )->where(
                                        'stock_location_code',
                                        '=',
                                        $stockIssue->stock_location_code
                                    )
                                    ->first();
                                $remainingQuantity = $stock->remaining_quantity - $stockHistory['quantity'];
                                $stock->update([
                                    'remaining_quantity' => $remainingQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow()
                                ]);
                                $this->stock_history_function->addStockHistory($stockHistory, $stockIssue->code, StockHistorySource::STOCK_ISSUE, $stock);

                            }
                        }
                    }
                }
                DB::commit();
                return response()->json([
                    'status' => 200,
                    'message' => "stockIssue Updated",

                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => "No Stock History provided",

                ], 200);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }


    public function destroy($id)
    {
        try {
            $stockIssue = StockIssue::select('id', 'is_active')->where('code', $id)->where('is_active', '=', 1)->first();
            $stockIssue->update([
                'is_active' => 0
            ]);
            return response()->json([
                'status' => 204,
                'message' => "stockIssue- Deleted"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }
}
