<?php

namespace App\Http\Controllers\Report_Location_Based_Controller;

use App\Http\Controllers\Controller;
use App\Models\Item_Models\ItemCategory;
use App\Models\Item_Models\ItemMaster;
use App\Models\Item_Models\ItemSubCategory;
use App\Models\Stock_Models\StockTransfer;
use Illuminate\Http\Request;
use Exception;
use App\Models\StockIssue_Models\StockIssue;
use App\Models\Stock_Models\Stock;
use Illuminate\Support\Facades\DB;

class StockLocationReportController extends Controller
{
    /**
     * stockIssueByLocation
     * logic: location based stock issue for customer
     * Author: Fathima Sajana
     * Date : 2023.03.02
     * version : 01
     * @param  mixed $request
     * @return void
     */
    public function stockIssueByLocation(Request $request)
    {
        $stockIssue = StockIssue::select(
            'code',
            'stock_location_code',
            'destination_code',
            'total',
            'created_at'
        )
            ->with([
                'stockLocation' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])

            ->with([
                'customer' => function ($query) {
                    $query->select('code', 'first_name')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'department' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories' => function ($query) {
                    $query->select('source_code', 'quantity', 'cost_price', 'retail_price')
                    ->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories.itemMaster' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])
            ->whereBetween('created_at', [date('Y-m-d H:m:s', strtotime($request->fromDate)), date('Y-m-d H:m:s', strtotime($request->toDate))])
            ->where('stock_location_code', '=', getCurrentLocationCode($request))
            ->get();
        return response()->json(['status' => 200, 'stockIssue' => $stockIssue]);
    }


    /**
     * stockTransferByLocation
     * logic: stock trnasfer
     * date: 2023.03.07
     * version: 01
     * author: fathima sajana
     * @param  mixed $request
     * @return void
     */
    public function stockTransferByLocation(Request $request)
    {
        $stockTransfer = StockTransfer::select(
            'code',
            'stock_location_code_from',
            'stock_location_code_to',
            'created_at'
        )
            ->with([
                'stockLocation' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])

            ->with([
                'stockHistories.itemMaster' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])
            ->whereBetween('created_at', [date('Y-m-d H:m:s', strtotime($request->fromDate)), date('Y-m-d H:m:s', strtotime($request->toDate))])
            ->where('stock_location_code_from', '=', getCurrentLocationCode($request))
            ->get();
        return response()->json(['status' => 200, 'stockTransfer' => $stockTransfer]);
    }


    public function itemInStockByLocation(Request $request)
    {
        try {
            $itemCode = $request->itemCode;
            $itemInStock = Stock::select('stock_location_code', 'item_code', 'remaining_quantity', 'stocks.cost_price', 'stocks.retail_price')
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'itemMaster' => function ($query) {
                        $query->select('code', 'barcode', 'name')->where('is_active', '=', 1);
                    }
                ])

                ->where('item_code', $itemCode)
                // ->groupBy('item_code')
                ->where('is_freeze', '', 0)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->get();


            return response()->json(['status' => 200, 'itemInStock' => $itemInStock]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }

    }

    /**
     * stockInHandTotalByLocation
     * Logic: Get the sum of the remaining quantity for a item based on locations.
     * Author: Suhail Jamaldeen
     * Date: 14.03.2023
     * Version: 01
     * @param  mixed $request
     * @return void
     */
    public function stockInHandTotalByLocation(Request $request)
    {
        try {
            $itemCode = $request->itemCode;
            $locationCode = getCurrentLocationCode($request);
            if ($itemCode) {
                $itemInStock = Stock::selectRaw('item_code,
                        stock_location_code, sum(remaining_quantity) as remaining_quantity')
                        ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')

                        ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'itemMaster' => function ($query) {
                            $query->select('code', 'barcode', 'name', 'description', 'category_code')->where('is_active', '=', 1);
                        }
                    ])

                    ->where('item_code', $itemCode)
                    ->where('stock_location_code', $locationCode) // Add this line
                    ->groupBy('item_code')
                    ->where('is_freeze', '', 0)
                    ->groupBy('stock_location_code')
                    ->get();

            } else {
                $itemInStock = Stock::selectRaw('item_code,
                            stock_location_code, sum(remaining_quantity) as remaining_quantity')
                    ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')

                            ->with([
                            'stockLocation' => function ($query) {
                                $query->select('code', 'name')->where('is_active', '=', 1);
                            }
                        ])
                        ->with([
                            'itemMaster' => function ($query) {
                                $query->select('code', 'barcode', 'name', 'description', 'category_code')
                                ->where('is_active', '=', 1);
                            }
                        ])

                        // ->where('item_code', $itemCode)
                        ->groupBy('item_code')
                        ->where('stock_location_code', $locationCode) // Add this line
                        ->orderBy('item_masters.category_code')
                        ->where('is_freeze', '', 0)
                        ->groupBy('stock_location_code')
                        ->get();
            }



            return response()->json([
                'status' => 200,
                'itemInStock' => $itemInStock
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function stockDashboard(Request $request)
{
        try {
        $itemCode = $request->itemCode;
        $locationCode = $request->locationCode;
        $batch = $request->batch;

        if($itemCode && $locationCode)
        {
            $stockInHand = Stock::select()
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories' => function ($query) {
                        $query->select('item_code','source_code', 'stock_code', 'source', 'quantity', 'cost_price','wholesale_price','retail_price','least_price')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'itemMaster.itemCategory' => function ($query) {
                        $query->select('code', 'name')
                        ->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'itemMaster.itemBrand' => function ($query) {
                        $query->select('code', 'name')
                        ->where('is_active', '=', 1);
                    }
                ])
                //->where('item_masters.is_service', 0)
                ->where('remaining_quantity', '>', 0)
                ->where('item_code', $itemCode)
                //->where('batch', $batch)
                ->where('stock_location_code', $locationCode)
                ->get();

            }

            elseif($itemCode)
            {
                $stockInHand = Stock::select()
                        ->with([
                            'stockLocation' => function ($query) {
                                $query->select('code', 'name')->where('is_active', '=', 1);
                            }
                        ])
                        ->with([
                            'stockHistories' => function ($query) {
                                $query->select('item_code','source_code', 'stock_code', 'source', 'quantity', 'cost_price', 'wholesale_price', 'retail_price', 'least_price')->where('is_active', '=', 1);
                            }
                        ])
                        ->with([
                            'itemMaster.itemCategory' => function ($query)  {
                                $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                            }
                        ])
                        ->with([
                            'itemMaster.itemBrand' => function ($query) {
                                $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                            }
                        ])
                        ->where('remaining_quantity', '>', 0)
                        ->where('item_code', $itemCode)
                        //->where('stock_location_code', $locationCode)
                        ->get();
            }
                elseif($locationCode)
                {
                    $stockInHand = Stock::selectRaw('stocks.code, stocks.cost_price, stocks.retail_price, stocks.wholesale_price, stocks.item_code, stocks.stock_location_code, sum(stocks.remaining_quantity) as remaining_quantity')
                    ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')
                    ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'itemMaster' => function ($query) {
                            $query->select('code', 'barcode', 'name', 'description', 'category_code', 'cost_price', 'wholesale_price', 'retail_price', 'least_price')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->groupBy('stocks.code', 'stocks.cost_price', 'stocks.retail_price', 'stocks.wholesale_price', 'stocks.item_code', 'stocks.stock_location_code')
                    ->where('stock_location_code', $locationCode)
                    ->get();

            }

                return response()->json([
                'status' => 200,
                'stockInHand' => $stockInHand,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
}

public function stockDashboardCategory(Request $request)
{
    try{
        $categoryCode = $request->categoryCode;
        $brandCode = $request->brandCode;
        $key = getCurrentLocationCode($request);

       if($categoryCode && $brandCode)
       {
        $category = ItemMaster::select('code', 'reference_code', 'barcode', 'name', 'description', 'category_code', 'sub_category_code', 'unit_code', 'brand_code', 'color_code', 'bin_location_code', 'wholesale_price', 'retail_price', 'dealer_price', 'least_price', 'cost_price', 'model', 'min_stock', 'max_stock', 'reorder_stock', 'expiry_notification_days', 'is_service')
        ->with([
            'stocks' => function ($query) {
                $query
            ->where('is_freeze', '', 0)
            ->where('is_active', '=', 1);
            }
        ])

        ->with([
            'stocks.stockHistories' => function ($query) {
                $query->select('item_code','source_code', 'stock_code', 'source', 'quantity', 'cost_price', 'wholesale_price', 'retail_price', 'least_price')
            ->where('is_freeze', '', 0)
            ->where('is_active', '=', 1);
            }
        ])
        ->where('category_code',$request->categoryCode)
        ->where('brand_code',$request->brandCode)
        ->where('is_active', '=', 1)
        // ->where(function ($query) use ($key) {
        //     $query
        //     ->whereHas(
        //             'stocks',
        //             function ($query) use ($key) {
        //                 $query->where('stock_location_code', '=', $key);
        //             }
        //         );
        //     })
        ->get();
       }


        elseif($categoryCode)
        {
            $category = ItemMaster::select('code', 'reference_code', 'barcode', 'name', 'description', 'category_code', 'sub_category_code', 'unit_code', 'brand_code', 'color_code', 'bin_location_code', 'wholesale_price', 'retail_price', 'dealer_price', 'least_price', 'cost_price', 'model', 'min_stock', 'max_stock', 'reorder_stock', 'expiry_notification_days', 'is_service')
            ->with([
                'stocks' => function ($query) use ($request){
                $query ->where('stock_location_code',  '=', getCurrentLocationCode($request))
            ->where('is_freeze', '', 0)
            ->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories' => function ($query) {
                    $query->select('item_code','source_code', 'stock_code', 'source','quantity', 'cost_price', 'wholesale_price', 'retail_price', 'least_price')
            ->where('is_freeze', '', 0)
            ->where('is_active', '=', 1);
                }
            ])
            ->where('category_code',$request->categoryCode)
            ->where('is_active', '=', 1)
            // ->where(function ($query) use ($key) {
            //     $query
            //     ->whereHas(
            //             'stocks',
            //             function ($query) use ($key) {
            //                 $query->where('stock_location_code', '=', $key);
            //             }
            //         );
            //     })
            ->get();
        }

        elseif($brandCode)
        {
            $category = ItemMaster::select('code', 'reference_code', 'barcode', 'name', 'description', 'category_code', 'sub_category_code', 'unit_code', 'brand_code', 'color_code', 'bin_location_code', 'wholesale_price', 'retail_price', 'dealer_price', 'least_price', 'cost_price', 'model', 'min_stock', 'max_stock', 'reorder_stock', 'expiry_notification_days', 'is_service')
            ->with([
                'stocks' => function ($query) use ($request) {
                $query ->where('stock_location_code',  '=', getCurrentLocationCode($request))
                ->where('is_freeze', '', 0)
                ->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories' => function ($query) {
                    $query->select('item_code','source_code', 'stock_code', 'source','quantity', 'cost_price', 'wholesale_price', 'retail_price', 'least_price')
                ->where('is_freeze', '', 0)
                ->where('is_active', '=', 1);
                }
            ])
            ->where('brand_code', $request->brandCode)
            //->where('category_code',$request->categoryCode)
            ->where('is_active', '=', 1)
            // ->where(function ($query) use ($key) {
            //     $query
            //     ->whereHas(
            //             'stocks',
            //             function ($query) use ($key) {
            //                 $query->where('stock_location_code', '=', $key);
            //             }
            //         );
            //     })
            ->get();
        }

        return response()->json(['status' => 200,
        'category' => $category
    ]);

    } catch (\Exception $e) {
        DB::rollBack();
        throw new Exception($e);
    }
}

public function updatePrice(Request $request, $id)
    {
        try {


            $wholesalePrice = $this->parsePrice($request->wholesalePrice);
            $retailPrice = $this->parsePrice($request->retailPrice);
            $costPrice = $this->parsePrice($request->costPrice);
            $itemCode = $request->itemCode;
            $channgeAllLocation = $request->allLocation;

            if($channgeAllLocation) {
                $stocks = Stock::where('item_code', $itemCode)->get();
                foreach( $stocks as  $stock ) {
                    $stock->update([
                        'wholesale_price' => $wholesalePrice,
                        'retail_price' => $retailPrice,
                        'cost_price' => $costPrice,
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow(),
                    ]);
                }
            }else {
                $stock = Stock::where('code', $id)->first();
                $stock->update([
                    'wholesale_price' => $wholesalePrice,
                    'retail_price' => $retailPrice,
                    'cost_price' => $costPrice,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

            }

            return response()->json([
                'status' => 200,
                'updatePrice' => $stock
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }
    private function parsePrice($price)
    {
        if ($price !== null) {
            // Remove commas and convert to a float
            return floatval(str_replace(',', '', $price));
        }

        return null;
    }
}








