<?php

namespace App\Http\Controllers\Report_Location_Based_Controller;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Enums\MultiPurposeStatus;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Invoice_Models\Invoice;
use App\Models\Invoice_Models\InvoiceExt;
use App\Models\Invoice_Models\SalesReturn;
use App\Models\Invoice_Models\InvoiceTemps;

class SalesLocationReportController extends Controller
{
    //
    /**
     * salesLocationReport
     * logic: daily and periodic sales location based
     * Author: Fathima Sajana
     * Date : 2023.03.02
     * version : 01
     * @param  mixed $request
     * @return void
     */
    public function salesLocationReport(Request $request)
    {
        try {

            $repCode = $request->salesRepCode;
            $fromDate = $request->fromDate;
            $toDate = $request->toDate;
            $payOffInvoices = $request->payOff;

            $invoice = Invoice::select(
                'code',
                'series_code',
                'stock_location_code',
                'invoice_number',
                'date_time',
                'customer_code',
                'discount_percentage',
                'bill_discount',
                'discount_amount',
                'discount_total',
                'paid_amount',
                'sub_total_amount',
                'total_amount',
                'paid_status',
                'status',
                'pay_off',
                DB::raw('total_amount - paid_amount as out_standing')
            )
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer' => function ($query) {
                        $query->select('code', 'first_name', 'second_name', 'phone_1')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories' => function ($query) {
                        $query->select('code', 'source_code', 'source', )->where('is_active', '=', 1);
                    }
                ])
                // ->whereBetween('date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate) - 1), date('Y-m-d H:m:s', strtotime($request->toDate))])
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->where('status', '<>', MultiPurposeStatus::REVERSED)


                ->when($fromDate && $toDate, function ($query) use ($fromDate, $toDate) {
                    $query->whereBetween('date_time', [$fromDate, $toDate]);
                })
                ->when($repCode, function ($query) use ($repCode) {
                    $query->whereHas('customer.route', function ($query) use ($repCode) {
                        $query->where('rep_code', '=', $repCode);
                    });
                });
                if($payOffInvoices){
                    $invoice =  $invoice->where('pay_off','>', 0);
                }
                $invoice = $invoice->get();

            return response()->json(['status' => 200, 'salesReport' => $invoice], 200);

        } catch (\Exception $e) {
            throw new Exception($e);
        }

    }

      //
    /**
     * save invooice report
     * logic: Periodically saved invoice
     * Author: ALA.Saneeth
     * Date : 2024.08.14
     * version : 01
     * @param  mixed $request
     * @return void
     */
    public function savedInvoiceReport(Request $request)
    {
        try {
            $customer = $request->customer;

            if($customer){
                $invoice = InvoiceTemps::select(
                    'code',
                    'series_code',
                    'stock_location_code',
                    'invoice_number',
                    'date_time',
                    'customer_code',
                    'discount_percentage',
                    'bill_discount',
                    'discount_amount',
                    'discount_total',
                    'sub_total_amount',
                    'total_amount',
                    'paid_status',
                    'status'
                )
                    ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'customer' => function ($query) {
                            $query->select('code', 'first_name', 'second_name', 'phone_1')->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'StockHistoryTemp' => function ($query) {
                            $query->select('code', 'source_code', 'source')->where('is_active', '=', 1);
                        }
                    ])
                    ->whereBetween('date_time', [date('Y-m-d H:i:s', strtotime($request->fromDate) - 1), date('Y-m-d H:i:s', strtotime($request->toDate))])
                    ->where('stock_location_code', '=', getCurrentLocationCode($request))
                    ->where('status', '<>', MultiPurposeStatus::REVERSED)
                    ->where('customer_code','=',$customer)
                    ->get();
            }else {
            $invoice = InvoiceTemps::select(
                'code',
                'series_code',
                'stock_location_code',
                'invoice_number',
                'date_time',
                'customer_code',
                'discount_percentage',
                'bill_discount',
                'discount_amount',
                'discount_total',
                'sub_total_amount',
                'total_amount',
                'paid_status',
                'status'
            )
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer' => function ($query) {
                        $query->select('code', 'first_name', 'second_name', 'phone_1')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'StockHistoryTemp' => function ($query) {
                        $query->select('code', 'source_code', 'source')->where('is_active', '=', 1);
                    }
                ])
                ->whereBetween('date_time', [date('Y-m-d H:i:s', strtotime($request->fromDate) - 1), date('Y-m-d H:i:s', strtotime($request->toDate))])
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->where('status', '<>', MultiPurposeStatus::REVERSED)
                ->get();
            }

            return response()->json(['status' => 200,
             'salesReport' => $invoice,
             'customer'=> $customer

            ], 200);

        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * dailySalesReturnLocationBased
     * logic: daily and periodic sales return location based
     * Author: Fathima Sajana
     * Date : 2023.03.02
     * version : 01
     * @param  mixed $request
     * @return void
     */
    public function dailySalesReturnLocationBased(Request $request)
    {

        try {
            $customerCode = $request->customer;
            $repCode = $request->salesRepCode;
            $fromDate = $request->fromDate;
            $toDate = $request->toDate;

            if($customerCode) {
                $salesReturnReport = SalesReturn::select(
                    'code',
                    'stock_location_code',
                    'customer_code',
                    'invoice_code',
                    'total_amount',
                    'paid_status'
                )
                    ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'customer' => function ($query) {
                            $query->select('code', 'first_name')
                                ->where('is_active', '=', 1);
                        }
                    ])

                    ->with([
                        'stockHistories.itemMaster' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])

                    ->whereBetween('return_date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate) - 1), date('Y-m-d H:m:s', strtotime($request->toDate))], )

                    ->where('stock_location_code', '=', getCurrentLocationCode($request))
                    ->where('status', '<>', '5')
                    ->where('customer_code', $customerCode)
                    ->get();
            }else {
                $salesReturnReport = SalesReturn::select(
                    'code',
                    'stock_location_code',
                    'customer_code',
                    'invoice_code',
                    'total_amount',
                    'paid_status'
                )
                    ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'customer' => function ($query) {
                            $query->select('code', 'first_name')
                                ->where('is_active', '=', 1);
                        }
                    ])

                    ->with([
                        'stockHistories.itemMaster' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])

                    // ->whereBetween('return_date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate) - 1), date('Y-m-d H:m:s', strtotime($request->toDate))], )
                    ->where('stock_location_code', '=', getCurrentLocationCode($request))
                    ->where('status', '<>', '5')

                    ->when($fromDate && $toDate, function ($query) use ($fromDate, $toDate) {
                        $query->whereBetween('return_date_time', [$fromDate, $toDate]);
                    })
                    ->when($repCode, function ($query) use ($repCode) {
                        $query->whereHas('customer.route', function ($query) use ($repCode) {
                            $query->where('rep_code', '=', $repCode);
                        });
                    })
                    ->get();
            }


            return response()->json([
                'status' => 200,
                'salesReturnReport' => $salesReturnReport
            ], 200);

        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * detailsSalesReportLocationBased
     * logic: details (what are the items have) of the sales location based
     * Author: Fathima Sajana
     * Date : 2023.03.07
     * version : 01
     *
     * * logic: details (what are the items have) of the sales location based with paid status(fully, partial, credit)
     * Author: Fathima Sajana
     * Date : 2023.11.17
     * version : 02
     * @param  mixed $request
     * @return void
     */
    public function detailsSalesReportLocationBased(Request $request)
    {
        try {
            $paidStatus = $request->paidStatus;
            $fromDate = date('Y-m-d H:m:s', strtotime($request->fromDate));
            $toDate = date('Y-m-d H:m:s', strtotime($request->toDate));

            $query = Invoice::select(
                'code', 'series_code', 'stock_location_code', 'invoice_number', 'date_time',
                'customer_code', 'discount_percentage', 'bill_discount', 'discount_amount',
                'discount_total', 'paid_amount', 'sub_total_amount', 'total_amount', 'paid_status'
            )
            ->with([
                'stockLocation' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'customer' => function ($query) {
                    $query->select('code', 'first_name', 'second_name', 'phone_1')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories.itemMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->whereBetween('date_time', [$fromDate, $toDate])
            ->where('stock_location_code', '=', getCurrentLocationCode($request))
            ->where('status', '<>', MultiPurposeStatus::REVERSED);

            if ($paidStatus) {
                $query->where('paid_status', '=', $paidStatus);
            }

            $invoice = $query->get();

            return response()->json(['status' => 200, 'salesReport' => $invoice], 200);

        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }



    /**
     * monthlySalesLocationBased
     * author: fathima sajana
     * date: 2023.04.02
     * logic: monthly sales location based
     * version: 01
     * @param  mixed $request
     * @return void
     */
    public function monthlySalesLocationBased(Request $request)
    {
        $month = $request->month;
        $year = $request->year;
        $stockLocationCode = getCurrentLocationCode($request);

        // Retrieve invoices and group by date
        $monthlySales = Invoice::with(['stockHistories' => function ($query) {
                $query->where('is_active', '=', 1)
                ->where('status', '!=', 5);

            }])
            ->select([
                DB::raw('SUM(total_amount) as invoice_total'), // Sum of total_amount for each date
                DB::raw('DATE(date_time) as date')             // Extract date part only
            ])
            ->where('stock_location_code', '=', $stockLocationCode)
            ->whereMonth('date_time', $month)
            ->whereYear('date_time', $year)
            ->groupBy(DB::raw('DATE(date_time)'))             // Group by date only
            ->get()
            ->map(function ($record) {
                $date = $record->date;
                $invoicesForDate = Invoice::with(['stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1)
                        ->where('status', '!=', 5);
                    }])
                    ->whereDate('date_time', $date)
                    ->get();

                // Calculate total profit for this date
                $totalProfit = $invoicesForDate->sum(function ($invoice) {
                    return $invoice->stockHistories->sum(function ($history) {
                        $profit = ($history->unitSoldPrice - $history->costPrice) * $history->quantity;
                        return $profit;
                    });
                });

                return [
                    'date' => $date,
                    'invoiceTotal' => $record->invoice_total,
                    'totalProfit' => $totalProfit,
                    'stockHistories' => $invoicesForDate->pluck('stockHistories')->flatten(1)
                ];
            });

        return response()->json(['status' => 200, 'monthlySales' => $monthlySales], 200);
    }
    /**
     * Author: Suhail Jamaldeen
     * Version: 02
     * Date: 13.10.2023
     * Logic: Get the sales for customer by date
     * Summary of itemsSalesByCustomer
     * @param \Illuminate\Http\Request $request
     * @throws \Exception
     * @return \Illuminate\Http\JsonResponse|mixed
     */
    public function itemsSalesByCustomer(Request $request)
    {
        try {
            $fromDate = strtotime($request->fromDate) - 1;
            $toDate = strtotime($request->toDate);
            $days = abs(round(($toDate - $fromDate) / 86400));
            $result = [];

            // for ($i = 0; $i < $days; $i++) {

            $fromDate = Carbon::parse($fromDate)->addDays(1);
            $toDate = Carbon::parse($fromDate)->addDays(100);
            $itemsSalesByCustomer = Invoice::select(
                'code',
                'stock_location_code',
                'invoice_number',
                'date_time',
                'customer_code',
                'discount_percentage',
                'bill_discount',
                'discount_amount',
                'discount_total',
                'paid_amount',
                'sub_total_amount',
                'total_amount',
                'paid_status'
            )
                ->with([
                    'stockLocation' => function ($query) {
                        $query->select(
                            'code',
                            'name'
                        )->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer' => function ($query) {
                        $query->select(
                            'code',
                            'first_name',
                            'second_name',
                            'phone_1'
                        )->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories' => function ($query) {
                        $query->select(
                            'stock_histories.code',
                            'stock_histories.source_code',
                            'stock_histories.source',
                            'stock_histories.quantity',
                            'stock_histories.item_code',
                            'item_masters.name'
                        )
                            ->join(
                                'item_masters',
                                'stock_histories.item_code',
                                '=',
                                'item_masters.code'
                            )
                            ->where('stock_histories.is_active', '=', 1);
                    }
                ])
                ->whereBetween(
                    'date_time',
                    [date('Y-m-d H:m:s', strtotime($request->fromDate) - 1), date('Y-m-d H:m:s', strtotime($request->toDate) + 1)]
                )

                ->where(
                    'stock_location_code',
                    '=',
                    getCurrentLocationCode($request)
                )
                ->get()
                // ->groupBy(function ($key) {
                //     return Carbon::parse($key->date)->format('Y-m-d'); // grouping by days
                // })
                // ->groupBy(function ($key) {
                //     return $key->customer_code; // grouping by days
                // })
                ;
            // if (sizeof($itemsSalesByCustomer) > 0) {
            //     $result[$i] = $itemsSalesByCustomer;
            // }

            // }


            return response()->json([
                'status' => 200,
                'itemsSalesByCustomer' => $itemsSalesByCustomer
            ], 200);

        } catch (\Exception $e) {
            throw new Exception($e);
        }}

    /**
     * consignBatch (for potlatch)
     * logic: batch wise invoices (paiid status 3 = fully settled, 2 = partially settled, 1 = credit)
     * date : 2023.06.12
     * Author: Fathima Sajana
     * version :01
     * @param  mixed $request
     * @return void
     */
    public function consignBatch(Request $request)
    {
        try {
            $batchCode = $request->batchCode;
            $paidStatus = $request->paidStatus;

            if ($batchCode && $paidStatus)
                $invoiceExt = InvoiceExt::select('code', 'invoice_code', 'batch_code', 'paid_status')
                    ->with([
                        'invoice' => function ($query) {
                            $query->select('code', 'total_amount', 'paid_amount', 'paid_status', 'customer_code')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'invoice.customer' => function ($query) {
                            $query
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'invoice.stockHistories.itemMaster' => function ($query) {
                            $query->where('is_active', '=', 1);
                        }
                    ])
                    ->where('batch_code', '=', $batchCode)
                    ->where('paid_status', '=', $paidStatus)
                    //->where('stock_location_code', '=', getCurrentLocationCode($request))
                    // ->whereHas('invoice.stockLocation', function ( $query) use ($request)
                    // {$query
                    // ->where('code', '=', getCurrentLocationCode($request));})
                    ->get();
            else {
                $invoiceExt = InvoiceExt::select('code', 'invoice_code', 'batch_code', 'paid_status')
                    ->with([
                        'invoice' => function ($query) {
                            $query->select('code', 'total_amount', 'paid_amount', 'paid_status', 'customer_code')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'invoice.customer' => function ($query) {
                            $query
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'invoice.stockHistories.itemMaster' => function ($query) {
                            $query->where('is_active', '=', 1);
                        }
                    ])
                    ->where('batch_code', '=', $batchCode)
                    // ->whereHas('invoice.stockLocation', function ( $query) use ($request)
                    // {$query
                    // ->where('code', '=', getCurrentLocationCode($request));})
                    ->get();
            }


            return response()->json(['status' => 200, 'consignBatch' => $invoiceExt], 200);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    public function seriesBasedInvoices(Request $request)
    {
        try {
            $seriesCode = $request->seriesCode;

            $seriesBasedInvoices = Invoice::select('code', 'series_code', 'stock_location_code', 'invoice_number', 'date_time', 'customer_code', 'discount_percentage', 'bill_discount', 'discount_amount', 'discount_total', 'paid_amount', 'sub_total_amount', 'total_amount', 'paid_status')
                ->with([
                    'invoiceExt' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                //->where('series_code', '=',  $seriesCode)
                ->whereBetween('date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)), date('Y-m-d H:m:s', strtotime($request->toDate))])
                ->whereBetween('series_code', [($request->from), ($request->to)])
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->get();

            return response()->json(['status' => 200, 'seriesBasedInvoices' => $seriesBasedInvoices], 200);

        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * show
     * Author : fathima sajana
     * date: 2023.11.20
     * version: 01
     * logic: particular invoice has multiple payment receipt, this is showing here
     * @param  mixed $id
     * @return void
     */
    public function customerSalesReconciliation(Request $request)
    {
        try {

            $paidStatus = $request->paidStatus;
            $fromDate = date('Y-m-d H:m:s', strtotime($request->fromDate));
            $toDate = date('Y-m-d H:m:s', strtotime($request->toDate));
            $customerCode = $request->customerCode;

                $customerSalesReconciliation = Invoice::select(
                    'code',
                    'invoice_number',
                    'invoice_source_code',
                    'date_time',
                    'customer_code',
                    'sales_quotation_code',
                    'sales_order_code',
                    'discount_amount',
                    'discount_percentage',
                    'bill_discount',
                    'discount_total',
                    'paid_amount',
                    'sub_total_amount',
                    'total_amount',
                    'paid_status',
                    'status',
                    'prints',
                    'received_amount',
                    'remarks',
                    'created_at',
                    'created_by'
                )
                ->with([
                    'customer.cusCheqReturnReceipt.cheqRtrnModeOfPayments' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'salesReturns' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'paymentReceipts' => function ($query) {
                        $query->where('payment_receipts.status', '<>', 5);
                    },
                    'paymentReceipts.modeOfPayments' => function ($query) {
                        //$query->where('payment_receipts.mode_of_payments.status', '<>', 5);
                    }
                ])

                ->where('invoices.is_active', '=', 1)
                ->where('status', '<>', 5)
                //->whereBetween('date_time', [$fromDate, $toDate])
                ->when($fromDate && $toDate, function ($query) use ($fromDate, $toDate) {
                    $query->whereBetween('date_time', [$fromDate, $toDate]);
                })
                ->when($paidStatus, function ($query) use ($paidStatus) {
                    $query->where('paid_status', '=', $paidStatus);
                })
                ->when($customerCode, function ($query) use ($customerCode) {
                    $query->where('customer_code', '=', $customerCode);
                })
                ->get();


            return response()->json([
                'status' => 200,
                'customerSalesReconciliation' => $customerSalesReconciliation,
            ]);
        } catch (Exception $e) {
            throw new Exception($e);
        }
    }


}
