<?php

namespace App\Http\Controllers\Report_Controllers;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Stock_Controllers\StockHistoryController;
use App\Models\Stock_Models\Stock;
use App\Models\Stock_Models\StockHistory;
use App\Models\StockIssue_Models\StockIssue;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockReportController extends Controller
{

    public function stockInHand(Request $request)
    {
        try {
            $locationCode = $request->locationCode;
            if ($locationCode) {
                $stockInHand = Stock::select(
                    'item_code',
                    'stock_location_code',
                    'remaining_quantity',
                    'item_masters.name',
                    'item_masters.barcode',
                    'item_masters.description',
                    'stocks.cost_price',
                    'stocks.retail_price'
                )
                    ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')
                    ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'itemMaster' => function ($query) {
                            $query->select('code', 'name', 'category_code','min_stock','max_stock')
                            ->where('is_active', '=', 1);
                        }
                    ])
                    ->where('item_masters.is_service', 0)
                    ->where('remaining_quantity', '>', 0)
                    ->where('stock_location_code', $locationCode)
                    ->where('is_freeze', '', 0)
                    ->orderBy('item_masters.category_code')
                    ->get();
            } else
            {
                $stockInHand = Stock::select('item_code', 'stock_location_code', 'remaining_quantity', 'item_masters.barcode', 'item_masters.name', 'item_masters.description', 'stocks.cost_price', 'stocks.retail_price')
                    ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')
                    ->with([
                        'stockLocation' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'itemMaster' => function ($query) {
                            $query->select('code', 'name', 'category_code','min_stock','max_stock')
                            ->where('is_active', '=', 1);
                        }
                    ])
                    ->where('remaining_quantity', '>', 0)
                    ->where('item_masters.is_service', 0)
                     ->where('is_freeze', '', 0)
                    ->orderBy('item_masters.category_code')
                    ->get();
            }
            return response()->json(['status' => 200, 'stockInHand' => $stockInHand]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }


    public function periodicStock(Request $request)
    {
        $key = getCurrentLocationCode($request);

        $periodicStock = StockHistory::select('code', 'item_code', 'stock_code', 'source_code', 'source', 'cost_price', 'retail_price', 'created_at', 'quantity', 'remaining_quantity')
            ->with([
                'itemMaster' => function ($query) {
                    $query->select('code', 'barcode', 'name', 'cost_price','description')->where('is_active', '=', 1);
                }
            ])->with([
                'grn.supplier' => function ($query) {
                    $query->select(
                        'code',
                        'name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stock' => function ($query) {
                    $query
                    ->select('code', 'stock_location_code', 'remaining_quantity')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stock.stockLocation' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])->with([
                'stockIn' => function ($query) {
                    $query->select(
                        'code',
                        'stock_location_code',
                        'reference_name'
                    )
                    ->where('is_active', '=', 1);
                }
            ])
            ->with([
                'invoice.customer' => function ($query) {
                    $query->select(
                        'code',
                        'first_name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'salesReturn' => function ($query) {
                    $query->select(
                        'code',
                        'customer_code',
                        'stock_location_code'
                    )->where('status', '<>', 5)
                    ->where('is_active', '=', 1);
                },
                'salesReturn.customer' => function ($query) {
                    // Only load the customer relationship when the salesReturn status is not equal to 5
                    $query->select(
                        'code',
                        'first_name',
                    );
                }
            ])
            ->with([
                'stockTransfer' => function ($query) use ($key) {
                    $query->select(
                        'code',
                        'stock_location_code_from',
                        'stock_location_code_to',
                    )->with([
                        'stockLocationFrom' => function ($query) {
                            $query->select('code', 'name');
                        },
                        'stockLocationTo' => function ($query) {
                            $query->select('code', 'name');
                        }
                    ])->where('is_active', '=', 1);
                }
            ])
            // ->with([
            //     'stockTransfer.stockLocation' => function ($query) {
            //         $query->select(
            //             'code',
            //             'stock_location_code_to',
            //         )->where('is_active', '=', 1);
            //     }
            // ])
            ->with([
                'stockIssue.customer' => function ($query) {
                    $query->select(
                        'code',
                        'first_name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'purchaseReturn.supplier' => function ($query) {
                    $query->select(
                        'code',
                        'name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->whereBetween('updated_at', [date('Y-m-d H:m:s', strtotime($request->fromDate)), date('Y-m-d H:m:s', strtotime($request->toDate))])
            ->where('status', '<>', 5)
            ->where('is_freeze', '', 0)
            ->get();
        return response()->json(['status' => 200, 'periodicStock' => $periodicStock]);

    }

    public function stockIssue(Request $request)
    {
        $stockIssue = StockIssue::select(
            'code',
            'destination_code',
            'total',
            'created_at'
        )
            ->with([
                'customer' => function ($query) {
                    $query->select('code', 'first_name')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'department' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories' => function ($query) {
                    $query->select('source_code', 'quantity', 'cost_price', 'retail_price')->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories.itemMaster' => function ($query) {
                    $query->select('code', 'name')->where('is_active', '=', 1);
                }
            ])
            ->whereBetween('created_at', [date('Y-m-d H:m:s', strtotime($request->fromDate)), date('Y-m-d H:m:s', strtotime($request->toDate))])
            ->get();
        return response()->json(['status' => 200, 'stockIssue' => $stockIssue]);

        //
    }


    public function stockSummary(Request $request)
    {
        $locationCode = $request->locationCode;

        $stockSummary = Stock::select(
            'item_code',
            'stock_location_code',
            'item_masters.name',
            'item_masters.description',
            'stocks.retail_price',
            'stocks.cost_price',
            (DB::raw('SUM(stocks.remaining_quantity)'))
        )
            ->with([
                'itemMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->where('item_masters.is_service', 0)
            ->get();


        //     $stockSummary = Stock::selectRaw(' SUM(stocks.remaining_quantity) as remaining_quantity')
        //      ->groupBy('item_code')
        //    ->get();

        //    $stockSummary = Stock::select(['item_code', 'location_code', 'item_masters.name', 'item_masters.description', 'item_masters.retail_price', 'item_masters.cost_price'])
        //    ->withCount([ function($query) {
        //         $query->select(DB::raw('SUM(stocks.remaining_quantity)'));
        //     }
        // ])->with(['itemMaster' => function ($query) {
        //                 $query->where('is_active', '=', 1);
        //             }])
        //             ->where('item_masters.is_service',0)->get();

        //    $stockSummary = Stock::select([
        //     'customers.id',
        //     'customers.last_name'
        // ])->withCount([
        //     'customerInvoices as invoice_sum' => function($query) {
        //         $query->select(DB::raw('SUM(total_price)'));
        //     }
        // ])->whereHas('customerInvoices', function(Builder $q) {
        //     $q->where('customer_invoices.status', 1);
        // })->get();


        return response()->json(['status' => 200, 'stockInHand' => $stockSummary]);
    }

    /**
     * stockAudit
     * Author: Suhail Jamaldeen
     * Version: 01
     * Date: 15.03.2023
     * Logic: Get the stock audi details based on the location. Complex where condition has used.
     * @param  mixed $request
     * @return void
     */
    public function stockAudit(Request $request)
    {
        $key = getCurrentLocationCode($request);

        $stockAudit = StockHistory::select(
            'code',
            'item_code',
            'source_code',
            'source',
            'cost_price',
            'retail_price',
            'created_at',
            'quantity',
            'remaining_quantity'
        )
            ->with([
                'itemMaster' => function ($query) {
                    $query->select(
                        'code',
                        'barcode',
                        'name',
                        'description',
                        'cost_price',
                        'retail_price'
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'grn.supplier' => function ($query) {
                    $query->select(
                        'code',
                        'name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockIn' => function ($query) {
                    $query->select(
                        'code',
                        'stock_location_code',
                        'reference_name'
                    )
                    ->where('is_active', '=', 1);
                }
            ])
            ->with([
                'invoice.customer' => function ($query) {
                    $query->select(
                        'code',
                        'first_name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'salesReturn' => function ($query) {
                    $query->select(
                        'code',
                        'customer_code',
                        'stock_location_code'
                    )->where('status', '<>', 5)
                    ->where('is_active', '=', 1);
                },
                'salesReturn.customer' => function ($query) {
                    // Only load the customer relationship when the salesReturn status is not equal to 5
                    $query->select(
                        'code',
                        'first_name',
                    );
                }
            ])
            ->with([
                'stockTransfer' => function ($query) use ($key) {
                    $query->select(
                        'code',
                        'stock_location_code_from',
                        'stock_location_code_to',
                    )->with([
                        'stockLocationFrom' => function ($query) {
                            $query->select('code', 'name');
                        },
                        'stockLocationTo' => function ($query) {
                            $query->select('code', 'name');
                        }
                    ])->where('is_active', '=', 1);
                }
            ])
            // ->with([
            //     'stockTransfer.stockLocation' => function ($query) {
            //         $query->select(
            //             'code',
            //             'stock_location_code_to',
            //         )->where('is_active', '=', 1);
            //     }
            // ])
            ->with([
                'stockIssue.customer' => function ($query) {
                    $query->select(
                        'code',
                        'first_name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->with([
                'purchaseReturn.supplier' => function ($query) {
                    $query->select(
                        'code',
                        'name',
                    )->where('is_active', '=', 1);
                }
            ])
            ->where('item_code', $request->itemCode)
            ->where('is_freeze', '', 0)
            ->where('status', '<>', 5)
            ->where(function ($query) use ($key) {
                $query
                ->orwhereHas(
                        'grn',
                        function ($query) use ($key) {
                            $query->where('stock_location_code', '=', $key);
                        }
                    )
                    ->orwhereHas(
                        'stockIn',
                        function ($query) use ($key) {
                            $query->where('stock_location_code', '=', $key);
                        }
                    )
                    ->orwhereHas(
                        'invoice',
                        function ($query) use ($key) {
                                $query->where('stock_location_code', '=', $key);
                            }
                    )
                    ->orwhereHas(
                        'salesReturn',
                        function ($query) use ($key) {
                                $query->where('stock_location_code', '=', $key);
                            }
                    )
                    ->orwhereHas(
                        'stockTransfer',
                        function ($query) use ($key) {
                                $query->where('stock_location_code_from', '=', $key);
                            }
                    )
                    ->orwhereHas(
                        'stockTransfer',
                        function ($query) use ($key) {
                                $query->where('stock_location_code_to', '=', $key);
                            }
                    )
                    ->orwhereHas(
                        'stockIssue',
                        function ($query) use ($key) {
                                $query->where('stock_location_code', '=', $key);
                            }
                    )
                    ->orwhereHas(
                        'purchaseReturn',
                        function ($query) use ($key) {
                                $query->where('stock_location_code', '=', $key);
                            }
                    );
            })
            ->get();

        return response()->json(
            [
                'status' => 200,
                'stockAudit' => $stockAudit
            ]
        );
    }


    public function itemInStock(Request $request)
    {
        try {
            $itemCode = $request->itemCode;
            if ($itemCode) {
                $itemInStock = Stock::select('stock_location_code', 'item_code', 'remaining_quantity', 'stock_locations.name', 'stocks.cost_price', 'stocks.retail_price')
                    ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')
                    ->join('stock_locations', 'stock_locations.code', '=', 'stocks.stock_location_code')
                    ->with([
                        'itemMaster' => function ($query) {
                            $query->select('code', 'barcode', 'name')->where('is_active', '=', 1);
                        }
                    ])

                    ->where('stock_locations.is_active', 1)
                    ->where('item_code', $itemCode)
                    ->get();
            } else {
                $itemInStock = Stock::select('stock_location_code', 'item_code', 'remaining_quantity', 'stock_locations.name', 'stocks.cost_price', 'stocks.retail_price')
                    ->join('item_masters', 'item_masters.code', '=', 'stocks.item_code')
                    ->join('stock_locations', 'stock_locations.code', '=', 'stocks.stock_location_code')
                    ->with([
                        'itemMaster' => function ($query) {
                            $query->select('code', 'barcode', 'name')->where('is_active', '=', 1);
                        }
                    ])
                    ->where('stock_locations.is_active', 1)
                    ->get();
            }

            return response()->json(['status' => 200, 'itemInStock' => $itemInStock]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }

    }
}
