<?php

namespace App\Http\Controllers\Report_Controllers;

use App\Enums\MultiPurposeStatus;
use App\Http\Controllers\Controller;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqReturnReceipt;
use App\Models\PaymentReceipt_Models\ChequePayment;
use Illuminate\Http\Request;
use Exception;


class ChequeReceiptReportController extends Controller
{
     /**
     * chequeByCustomer
     * author : fathima sajana
     * date : 2023.02.01
     * purpose: get all cheques by particular customer via many receipts
     * @param  mixed $request
     * @return void
     */
    public function chequeByCustomer(Request $request)
    {
        try {
            $key = $request->key;
            //$code = $request->code;

            $chequeByCustomer = ChequePayment::select('cheque_payments.code', 'cheque_payments.mode_of_payment_code', 'cheque_payments.branch_code', 'cheque_payments.number', 'cheque_payments.amount', 'cheque_payments.deposit_date', 'cheque_payments.bank_code', 'cheque_payments.created_at')
                ->with([
                    'bank' => function ($query) {
                        $query->select('code', 'name')
                            ->where('is_active', '=', 1);
                    },
                ])

                ->with([
                    'modeOfPayment' => function ($query) {
                        $query->select('code', 'payment_receipt_code')
                            ->where('is_active', '=', 1);
                    },
                    'modeOfPayment.paymentReceipt' => function ($query) {
                        $query->select('code', 'customer_code')
                            ->where('is_active', '=', 1);
                    },
                    'modeOfPayment.paymentReceipt.customer' => function ($query) {
                        $query->select('code', 'first_name')
                            ->where('is_active', '=', 1);
                    }
                ])
                ->whereHas('modeOfPayment.paymentReceipt.customer', function ( $query) use ($key)
                    {$query
                        ->where('first_name', 'LIKE', '%' . $key . '%');})

                ->orwhereHas('modeOfPayment.paymentReceipt.customer', function ( $query) use ($key)
                    {$query
                        ->where('code', 'LIKE', '%' . $key . '%');})
                ->get();

            return response()->json([
                'status' => 200,
                'chequeByCustomer' => $chequeByCustomer
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * receiptChequeInHand
     * Author: Suhail Jamaldeen
     * Date: 01.02.2023
     * Version: 01
     * Logic: Get the cheques in hands received via receipts.
     *
     * author: fathima sajana
     * Date: 2023.04.02
     * Version: 02
     * Logic: location based cheques in hands received via receipts .
     * @param  mixed $request
     * @return void
     */
    public function periodicCheque(Request $request)
    {
        try {
            $periodicCheque = ChequePayment::select(
                'cheque_payments.code',
                'cheque_payments.mode_of_payment_code',
                'cheque_payments.bank_code',
                'cheque_payments.branch_code',
                'cheque_payments.number',
                'cheque_payments.deposit_date',
                'cheque_payments.created_at',
                'cheque_payments.amount',
                'cheque_payments.status',
                'cheque_payments.date_time'

            )
                ->with([
                    'bank' => function ($query) {
                        $query->select('code', 'name')
                            ->where('is_active', '=', 1);
                    },
                ])
                ->with([
                    'modeOfPayment.paymentReceipt.customer' => function ($query) {
                        $query->select('code', 'first_name')
                            ->where('is_active', '=', 1);
                    },
                ])

                ->whereBetween('cheque_payments.date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))])
                ->whereHas('modeOfPayment.paymentReceipt.stockLocation', function ( $query) use ($request)
                {$query
               ->where('code', '=', getCurrentLocationCode($request));})
                ->get();

            return response()->json([
                'status' => 200,
                'receiptChequeInHand' => $periodicCheque
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

public function chequeInHand(Request $request)
{
    try {

        $customerCode = $request->code;

        $chequeInHand = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.customer_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.created_at',
            'cheque_payments.amount',
            'cheque_payments.status',
            'cheque_payments.date_time'

        )
        ->with([
            'modeOfPayment.paymentReceipt.customer' => function ($query) {
                $query->select('code', 'first_name')
                    ->where('is_active', '=', 1);
            },
        ])
        ->where('cheque_payments.customer_code', '=', $customerCode)
        ->where('cheque_payments.status', '=', 6)
        ->whereBetween('cheque_payments.date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))])
        ->get();

        return response()->json([
            'status' => 200,
            'chequeInHand' => $chequeInHand
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 500,
            'message' => $e
        ], 500);
    }

}

public function chequeByRep(Request $request)
{
   // try {
        $customerCode = $request->code;
        $salesRep = $request->salesRep;
        $chequeStatus = $request->chequeStatus;

        $chequeInHand = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.customer_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.created_at',
            'cheque_payments.amount',
            'cheque_payments.status',
            'cheque_payments.date_time'
        )
        ->with([
            'bank' => function ($query) {
                $query->where('is_active', '=', 1);

            },
        ])
        ->with([
            'modeOfPayment.paymentReceipt.customer' => function ($query) {
                $query->select('code', 'first_name')
                    ->where('is_active', '=', 1);
            },
            'modeOfPayment.paymentReceipt.customer.route.salesRep' => function ($query) {
                $query->select('code', 'username')
                    ->where('is_active', '=', 1);
            }

        ])

       // ->where('cheque_payments.customer_code', '=', $customerCode)
        ->whereBetween('cheque_payments.date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))]);
        //->get();

        if ($chequeStatus) {
            $chequeInHand = $chequeInHand->where('cheque_payments.status', '=', $chequeStatus);
        }


        if ($salesRep) {
            $chequeInHand = $chequeInHand->whereHas('modeOfPayment.paymentReceipt.customer.route.salesRep', function ($query) use ($salesRep) {
                $query->where('code', '=', $salesRep);
            });
        }
       $chequeInHand = $chequeInHand->get();

        return response()->json([
            'status' => 200,
            'chequeByRep' => $chequeInHand,
            'chequeStatus'=>$chequeStatus
        ]);
    // } catch (\Exception $e) {
    //     return response()->json([
    //         'status' => 500,
    //         'message' => $e
    //     ], 500);
    // }
}


public function chequeInProgress(Request $request)
{
    try {

        $customerCode = $request->code;

        $chequeInProgress = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.customer_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.created_at',
            'cheque_payments.amount',
            'cheque_payments.status',
            'cheque_payments.date_time'

        )
        ->with([
            'modeOfPayment.paymentReceipt.customer' => function ($query) {
                $query->select('code', 'first_name')
                    ->where('is_active', '=', 1);
            },
        ])
        ->where('cheque_payments.customer_code', '=', $customerCode)
        ->where('cheque_payments.status', '=', 7)
        ->whereBetween('cheque_payments.date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))])
        ->get();

        return response()->json([
            'status' => 200,
            'chequeInProgress' => $chequeInProgress
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 500,
            'message' => $e
        ], 500);
    }

}


public function chequeInRealized(Request $request)
{
    try {
        $customerCode = $request->code;
        $repCode = $request->repCode;
        $fromDate = $request->fromDate;
        $toDate = $request->toDate;

        // Start building the query
        $chequeInRealized = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.customer_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.created_at',
            'cheque_payments.amount',
            'cheque_payments.status',
            'cheque_payments.date_time',
            'cheque_payments.updated_at'
        )
        ->with([
            'modeOfPayment.paymentReceipt.customer' => function ($query) {
                $query->select('code', 'first_name')
                    ->where('is_active', '=', 1);
            },
        ])
        ->with([
            'modeOfPayment.paymentReceipt.customer.route.salesRep' => function ($query) {
                $query->where('is_active', '=', 1);
            },
        ])

        ->where('status', '=', 11)
        ->when($fromDate && $toDate, function ($query) use ($fromDate, $toDate) {
            $query->whereBetween('updated_at', [$fromDate, $toDate]);
        })
        ->when($repCode, function ($query) use ($repCode) {
            $query->whereHas('modeOfPayment.paymentReceipt.customer.route', function ($query) use ($repCode) {
                $query->where('rep_code', '=', $repCode);
            });
        })
        ->when($customerCode, function ($query) use ($customerCode) {
            $query->where('customer_code', '=', $customerCode);
        })

        ->get();

        return response()->json([
            'status' => 200,
            'chequeInRealized' => $chequeInRealized
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 500,
            'message' => $e->getMessage()
        ], 500);
    }
}





public function chequeRetured(Request $request)
{
    try {

        $customerCode = $request->code;
        $repCode = $request->salesRepCode;
        $fromDate = $request->fromDate;
        $toDate = $request->toDate;

        $chequeRetured = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.customer_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.created_at',
            'cheque_payments.amount',
            'cheque_payments.status',
            'cheque_payments.date_time',
            'cheque_payments.updated_at'

        )
        ->with([
            'modeOfPayment.paymentReceipt.customer' => function ($query) {
                $query->select('code', 'first_name')
                    ->where('is_active', '=', 1);
            },
        ])
        ->with([
            'modeOfPayment.paymentReceipt.customer.route.salesRep' => function ($query) {
                $query->where('is_active', '=', 1);
            },
        ])
        ->where('cheque_payments.status', '=', 8)
        // ->whereBetween('cheque_payments.date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))])
        ->when($fromDate && $toDate, function ($query) use ($fromDate, $toDate) {
            $query->whereBetween('updated_at', [$fromDate, $toDate]);
        })
        ->when($repCode, function ($query) use ($repCode) {
            $query->whereHas('modeOfPayment.paymentReceipt.customer.route', function ($query) use ($repCode) {
                $query->where('rep_code', '=', $repCode);
            });
        })
        ->when($customerCode, function ($query) use ($customerCode) {
            $query->where('customer_code', '=', $customerCode);
        })

        ->get();

        return response()->json([
            'status' => 200,
            'chequeReturned' => $chequeRetured
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 500,
            'message' => $e
        ], 500);
    }

}

  /**
     * customerChequeReturnPayment
     * Author: Aasif Saneeth
     * Date: 19.04.2024
     * Version: 01
     * Logic: Get daily customer cheque return payments.
     * @param  mixed $request
     * @return void
     */
    public function customerChequeReturnPayment(Request $request)
    {
         try {
            $returnReceipts = CusCheqReturnReceipt::select(
                'code',
                'customer_code',
                'stock_location_code',
                'total_amount',
                'settled_amount',
                'over_payment',
                'status',
                'payment_date_time',
                'remarks',
                'created_by'
            )
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'cheqRtrnModeOfPayments' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'user' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->where('status', '<>', MultiPurposeStatus::REVERSED)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->whereBetween('payment_date_time', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))])
                ->get();

            return response()->json([
                'status' => 200,
                'returnReceipts' => $returnReceipts
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }
}
