<?php

namespace App\Http\Controllers\Payment_Controllers;

use App\Enums\ChequeStatusCode;
use App\Enums\DefaultValues;
use App\Enums\MultiPurposeStatus;
use App\Enums\PaymentStatus;
use App\Enums\PaymentTypeCode;
use App\Enums\TransactionCode;
use App\Http\Controllers\Controller;
use App\Models\Customer_Models\Customer;
use App\Models\CustomerChequeReturnPayment_Models\CheqPayCustCheqReceipt;
use App\Models\CustomerChequeReturnPayment_Models\CheqRtrnModeOfPayment;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqRtrnBankPayment;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqRtrnCardPayment;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqRtrnCashPayment;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqRtrnChequePayment;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqReturnReceipt;
use App\Models\CustomerChequeReturnPayment_Models\CusCheqRtrnOnlinePayment;
use App\Models\PaymentReceipt_Models\ChequePayment;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CustomerChequeReturnReceiptController extends Controller
{
    public function index(Request $request)
    {
        try {
            $returnReceipts = CusCheqReturnReceipt::select(
                'code',
                'customer_code',
                'stock_location_code',
                'total_amount',
                'settled_amount',
                'over_payment',
                'status',
                'payment_date_time',
                'remarks',
                'created_by'
            )
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'cheqRtrnModeOfPayments' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'user' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('created_at', 'DESC')
                ->paginate(DefaultValues::PAGINATION_VALUE);

            return response()->json([
                'status' => 200,
                'returnReceipts' => $returnReceipts
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function show($id)
    {
        try {
            $returnReceipt = CusCheqReturnReceipt::select(
                'code',
                'customer_code',
                'stock_location_code',
                'total_amount',
                'settled_amount',
                'over_payment',
                'status',
                'payment_date_time',
                'remarks',
                'created_by'
            )
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'cheqRtrnModeOfPayments' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'user' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('code', $id)
                ->where('is_active', '=', 1)
                ->first();
            if ($returnReceipt) {
                $cash = CusCheqRtrnCashPayment::select('amount')
                    ->where('cheque_receipt_code', $returnReceipt->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $cheque = CusCheqRtrnChequePayment::select('number', 'bank_code', 'branch_code', 'deposit_date', 'amount')
                ->with([
                    'bank' => function ($query) {
                        $query->select('code','name')
                        ->where('is_active', '=', 1);
                    }
                ])
                ->where('cheque_receipt_code', $returnReceipt->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $bank = CusCheqRtrnBankPayment::select('bank_code', 'reference_number', 'date', 'amount')
                ->with([
                    'bank' => function ($query) {
                        $query->select('code','name')
                        ->where('is_active', '=', 1);
                    }
                ])                
                ->where('cheque_receipt_code', $returnReceipt->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $onlinePayment = CusCheqRtrnOnlinePayment::select('bank_code', 'transfer_id', 'transfer_date', 'amount')
                ->with([
                    'bank' => function ($query) {
                        $query->select('code','name')
                        ->where('is_active', '=', 1);
                    }
                ])         
                ->where('cheque_receipt_code', $returnReceipt->code)
                    ->where('is_active', '=', 1)
                    ->get();    

                $card = CusCheqRtrnCardPayment::select('bank_code', 'card_number', 'amount')
                ->with([
                    'bank' => function ($query) {
                        $query->select('code','name')
                        ->where('is_active', '=', 1);
                    }
                ])
                ->where('cheque_receipt_code', $returnReceipt->code)
                    ->where('is_active', '=', 1)
                    ->get();


                $chequeDetails = CheqPayCustCheqReceipt::select(
                    'code',
                    'cheque_total',
                    'paid_amount',
                    'outstanding',
                    'cheque_receipt_code',
                    'cheque_paym_code'
                )
                    ->with([
                        'chequePayment' => function ($query) {
                            $query->where('is_active', '=', 1);
                        }
                    ])
                    ->where('cheque_receipt_code', $returnReceipt->code)
                    ->where('is_active', '=', 1)
                    ->get();

                return response()->json([
                    'status' => 200,
                    'returnReceipt' => $returnReceipt,
                    'cashPayment' => $cash,
                    'chequePayment' => $cheque,
                    'bankPayment' => $bank,
                    'onlinePayment' => $onlinePayment,
                    'cardPayment' => $card,
                    'chequeDetails' => $chequeDetails
                ]);
            } else {
                return response()->Json([
                    'status' => 204,
                    'message' => 'Cheque return payment receipt code ' . $id . ' not found',
                ]);
            }
        } catch (\Exception $e) {
            throw new Exception($e);
            // return response(['status' => 500, 'message' => $e]);
        }
    }

   
    
    public function returnChequesForCustomer($key)
    {
        try {
            $returnChequePayments = ChequePayment::select('cheque_payments.code', 'cheque_payments.receipt_code', 'cheque_payments.customer_code', 'cheque_payments.mode_of_payment_code', 'cheque_payments.amount', 'cheque_payments.status', 'cheque_payments.paid_amount')
                ->join('customers', 'customers.code', '=', 'cheque_payments.customer_code')
                ->where(function ($query) use ($key) {
                    $query->where('cheque_payments.customer_code', 'like', "%$key%");
                })
                ->where(function ($query) {
                    $query->where('cheque_payments.status', '=', '8')
                        ->orWhere('cheque_payments.status', '=', '12');
                })
                ->whereColumn('cheque_payments.amount', '>', 'cheque_payments.paid_amount')
                ->get();
            
            return response()->json(['status' => 200, 'returnChequePayments' => $returnChequePayments]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }
    

    /**
     * store
     * logic: cheque return receipt like a payment receipt. this settle for return cheques
     * version: 01
     * author: fathima sajana
     * date :2023.03.20
     * @param  mixed $request
     * @return void
     */
    public function store(Request $request)
    {
        //try 
        {
            DB::beginTransaction();
            $payment_code = TransactionCode::CHEQUE_RETURN_PAYMENT_RECEIPT_CODE;

            $stockLocationCode = getCurrentLocationCode($request);
            $locationPrefix = substr("$stockLocationCode", -2);
            $max_code = CusCheqReturnReceipt::select('code')
                ->where('stock_location_code', '=', $stockLocationCode)
                ->max('code');

            $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;


            $returnReceipt = CusCheqReturnReceipt::create([
                'code' => $payment_code . $locationPrefix . $max_id,
                'customer_code' => $request['customer']['code'] ?? null,
                'stock_location_code' => getCurrentLocationCode($request),
                'total_amount' => $request->totalAmount,
                'settled_amount' => $request->settledAmount,
                'over_payment' => ($request->totalAmount - $request->settledAmount),
                //Includes the previous over payment as well
                'status' => $request->payment['paidStatus'],
                'remarks' => $request->remarks,
                'payment_date_time' =>$request->paymentDateTime,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
            ]);
            $chequeReturnModeOfPayments = $request->modeOfPayments;
            $cheque = $request->selectedCheque;

            if ($returnReceipt) {

                if (count($chequeReturnModeOfPayments) > 0) {
                    foreach ($chequeReturnModeOfPayments as $cheque_return_mode_of_payment) {
                        $modePayment_code = TransactionCode::CHEQUE_RETURN_MODE_OF_PAYMENT_CODE;
                        $max_code = CheqRtrnModeOfPayment::max('code');
                        $max_id = $max_code == null ? config('global.code_value') + 1 : substr("$max_code", 3) + 1;

                        $chequeReturnModeOfPayment = CheqRtrnModeOfPayment::create([
                            'code' => $modePayment_code . $max_id,
                            'cheque_receipt_code' => $returnReceipt->code,
                            'payment_type_code' => $cheque_return_mode_of_payment['paymentType'],
                            'amount' => $cheque_return_mode_of_payment['amount'],
                            'created_by' => getUserCode(),
                            'created_at' => getDateTimeNow(),
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                        if ($cheque_return_mode_of_payment) {
                            if ($chequeReturnModeOfPayment->payment_type_code == PaymentTypeCode::CASH_PAYMENT_CODE) {
                                $this->addCashPayment($cheque_return_mode_of_payment, $chequeReturnModeOfPayment->code, $returnReceipt);
                            } else if ($chequeReturnModeOfPayment->payment_type_code == PaymentTypeCode::CHEQUE_PAYMENT_CODE) {
                                $this->addChequePayment($cheque_return_mode_of_payment, $chequeReturnModeOfPayment->code, $returnReceipt);
                            } elseif ($chequeReturnModeOfPayment->payment_type_code == PaymentTypeCode::BANK_PAYMENT_CODE) {
                                $this->addBankPayment($cheque_return_mode_of_payment, $chequeReturnModeOfPayment->code, $returnReceipt);
                            } elseif ($chequeReturnModeOfPayment->payment_type_code == PaymentTypeCode::CARD_PAYMENT_CODE) {
                                $this->addCardPayment($cheque_return_mode_of_payment, $chequeReturnModeOfPayment->code, $returnReceipt);
                            } elseif ($chequeReturnModeOfPayment->payment_type_code == PaymentTypeCode::ONLINE_TRANSFER_PAYMENT_CODE) {
                                $this->addOnlineTransferPayment($cheque_return_mode_of_payment, $chequeReturnModeOfPayment->code, $returnReceipt);
                            }
                        }
                    }

                    $customer = Customer::where('code', $returnReceipt->customer_code)->first();
                    if ($customer) {
                        $customer->update([
                            'cheque_return_amount' => $customer['cheque_return_amount'] - $returnReceipt['total_amount'],
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                    }
                }

                $selectedCheques = $request->returnedCheques;
                foreach ($selectedCheques as $selectedCheque) {
                    $existingCheque = ChequePayment::where('code', $selectedCheque['code'])->first();
                    $this->addChequePaymentForChequeReturnReceipt($returnReceipt, $selectedCheque, $existingCheque);

                    //Update the invoice table to identify how much is paid for the invoice value
                    if ($existingCheque) 
                    {
                        $paidStatus = PaymentStatus::PAYMENT_PARTIAL;
                        $totalPaidAmount = $existingCheque['paid_amount'] + $selectedCheque['paymentAmount'];
                        
                        if ($totalPaidAmount >= $existingCheque['amount']) 
                        {
                            $paidStatus = PaymentStatus::PAYMENT_COMPLETED;
                            $existingCheque->update([
                                'status' => ChequeStatusCode::CHEQUE_REALIZED,                              
                            ]);
                        }
                        $existingCheque->update([
                            'paid_amount' => $totalPaidAmount,
                            //'status' => ChequeStatusCode::RETURNED_CHEQUE_SETTLED,
                            'paid_status' => $paidStatus,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow(),
                        ]);
                    }
                }
            }
            DB::commit();
            return response()->json([
                'status' => 200,
                'message' => 'Cheque Return Receipt Created'
            ]);
        } 
        // catch (\Exception $e) {
        //     DB::rollBack();
        //     throw new Exception($e);
        // }
    }
    
    /**
     * addCashPayment
     * logic: cheque return receipt time we can use for another payment(for settlement) using cash, cheque, card, bank
     * version: 01
     * author: fathima sajnana
     * date: 2023.03.20
     * @param  mixed $chequeReturnModeOfPayment
     * @param  mixed $chequeReturnModeOfPaymentCode
     * @param  mixed $returnReceipt
     * @return void
     */
    public function addCashPayment($chequeReturnModeOfPayment, $chequeReturnModeOfPaymentCode, $returnReceipt)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_RETURN_CASH_PAYMENT_CODE;
            $cus_code = CusCheqRtrnCashPayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $cashPayment = CusCheqRtrnCashPayment::create([
                'code' => $payment_code . $max_id,
                'cheque_receipt_code' => $returnReceipt['code'],
                'mode_of_payment_code' => $chequeReturnModeOfPaymentCode,
                'amount' => $chequeReturnModeOfPayment['amount'],
                'date_time' => $returnReceipt['payment_date_time'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),

            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function addChequePayment($chequeReturnModeOfPayment, $chequeReturnModeOfPaymentCode, $returnReceipt)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_RETURN_CHEQUE_PAYMENT_CODE;
            $cus_code = CusCheqRtrnChequePayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;


            $cheque_Payment = CusCheqRtrnChequePayment::create([
                'code' => $payment_code . $max_id,
                'cheque_receipt_code' => $returnReceipt['code'],
                'mode_of_payment_code' => $chequeReturnModeOfPaymentCode,
                'bank_code' => $chequeReturnModeOfPayment['chequePayment']['bank'],
                'branch_code' => $chequeReturnModeOfPayment['chequePayment']['branchCode'],
                //'customer_code' => $invoice['customer']['code'] ?? null,
                'number' => $chequeReturnModeOfPayment['chequePayment']['chequeNumber'],
                'deposit_date' => $chequeReturnModeOfPayment['chequePayment']['chequeDate'],
                'amount' => $chequeReturnModeOfPayment['amount'],
                'date_time' => $returnReceipt['payment_date_time'],
                'remarks' => $chequeReturnModeOfPayment['chequePayment']['remarks'] ?? null,
                'status' => ChequeStatusCode::CHEQUE_IN_HAND,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function addBankPayment($chequeReturnModeOfPayment, $chequeReturnModeOfPaymentCode, $returnReceipt)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_RETURN_BANK_PAYMENT_CODE;
            $cus_code = CusCheqRtrnBankPayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $bankPayment = CusCheqRtrnBankPayment::create([
                'code' => $payment_code . $max_id,
                'cheque_receipt_code' => $returnReceipt['code'],
                'mode_of_payment_code' => $chequeReturnModeOfPaymentCode,
                'bank_code' => $chequeReturnModeOfPayment['bankDeposit']['bank'],
                //'bank_code' => null,
                //$modeOfPayment['bank']['code'],
                'reference_number' => $chequeReturnModeOfPayment['bankDeposit']['refNumber'],
                'amount' => $chequeReturnModeOfPayment['amount'],
                'date_time' => $returnReceipt['payment_date_time'],
                'date' => $chequeReturnModeOfPayment['bankDeposit']['date'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
            return response()->Json(['status' => 200, 'message' => 'Bank Payment Updated']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }


    public function addOnlineTransferPayment($chequeReturnModeOfPayment, $chequeReturnModeOfPaymentCode, $returnReceipt)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_RETURN_ONLINE_TRANSFER_PAYMENT_CODE;
            $cus_code = CusCheqRtrnOnlinePayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $cusCheqRtrnOnlinePayment = CusCheqRtrnOnlinePayment::create([
                'code' => $payment_code . $max_id,
                'cheque_receipt_code' => $returnReceipt['code'],
                'mode_of_payment_code' => $chequeReturnModeOfPaymentCode,
                'bank_code' => $chequeReturnModeOfPayment['onlineTransfer']['bank'],
                //'bank_code' => null,
                //$modeOfPayment['bank']['code'],
                'transfer_id' => $chequeReturnModeOfPayment['onlineTransfer']['transferId']?? null,
                'amount' => $chequeReturnModeOfPayment['amount'],
                'date_time' => $returnReceipt['payment_date_time'],
                'transfer_date' => $chequeReturnModeOfPayment['onlineTransfer']['transferDate'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
            return response()->Json(['status' => 200, 'message' => 'Payment Updated']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function addCardPayment($chequeReturnModeOfPayment, $chequeReturnModeOfPaymentCode, $returnReceipt)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_RETURN_CARD_PAYMENT_CODE;
            $cus_code = CusCheqRtrnCardPayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $CardPayment = CusCheqRtrnCardPayment::create([
                'code' => $payment_code . $max_id,
                'cheque_receipt_code' => $returnReceipt['code'],
                'mode_of_payment_code' => $chequeReturnModeOfPaymentCode,
                'bank_code' => $chequeReturnModeOfPayment['cardPayment']['bank'],
                'card_number' => $chequeReturnModeOfPayment['cardPayment']['cardNumber'],
                'amount' => $chequeReturnModeOfPayment['amount'],
                'date_time' => $returnReceipt['payment_date_time'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),

            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }
    
    /**
     * addChequePaymentForChequeReturnReceipt
     * logic: intermediate table for customer cheque return receipt and cheque payment table
     * author : fathima sajana
     * version: 01
     * date :2023.03.20
     * @param  mixed $returnReceipt
     * @param  mixed $selectedCheques
     * @param  mixed $existingCheque
     * @return void
     */
    public function addChequePaymentForChequeReturnReceipt($returnReceipt, $selectedCheque, $existingCheque)
    {
        try 
        {
            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_PAYMENT_CUSTOMER_CHEQUE_RECEIPT_CODE;
            $cus_code = CheqPayCustCheqReceipt::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $chequePaymentCustomerChequeReceipt = CheqPayCustCheqReceipt::create([
                'code' => $payment_code . $max_id,
                'cheque_receipt_code' => $returnReceipt->code,
                'cheque_paym_code' => $selectedCheque['code'],
                'paid_amount' => $selectedCheque['paymentAmount'],
                'cheque_total' => $selectedCheque['amount'],
                'outstanding' => $selectedCheque ? ($selectedCheque['amount'] - ($existingCheque['paid_amount'] + $selectedCheque['paymentAmount'])) : ($selectedCheque['amount']),
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);

            DB::commit();
        } 
        catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function reversal(Request $request)
    {

        try {
            $CusCheqReturnReceiptCode = $request->transactionCode;

            $returnReceipt = CusCheqReturnReceipt::where('code', '=', $CusCheqReturnReceiptCode)
                ->where('status', '<>', MultiPurposeStatus::REVERSED)
                ->where('is_active', '=', 1)->first();

            //If return Receipt found in the table
            if ($returnReceipt) {
                DB::beginTransaction();

                $returnReceipt->update([
                    'status' => MultiPurposeStatus::REVERSED,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);


                $modeOfPayment = CheqRtrnModeOfPayment::where('cheque_receipt_code', '=', $CusCheqReturnReceiptCode)
                    ->where('is_active', '=', 1)->get();
              

                if (count($modeOfPayment) > 0) {
                    foreach ($modeOfPayment as $mode_of_payment) {
                        $mode_of_payment->update([
                            'status' => MultiPurposeStatus::REVERSED,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);

                        $paymentType = $mode_of_payment->payment_type_code;
                        switch ($paymentType) {
                            case PaymentTypeCode::CASH_PAYMENT_CODE:
                                CusCheqRtrnCashPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::CARD_PAYMENT_CODE:
                                CusCheqRtrnCardPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::CHEQUE_PAYMENT_CODE:
                                CusCheqRtrnChequePayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::BANK_PAYMENT_CODE:
                                CusCheqRtrnBankPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::ONLINE_TRANSFER_PAYMENT_CODE:
                                CusCheqRtrnOnlinePayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;
                            default:
                                # code...
                                break;
                        }
                    }


                    $customer = Customer::where('code', $returnReceipt->customer_code, $mode_of_payment->cheque_receipt_code)->first();
                    if ($customer) {
                        $customer->update([
                            'cheque_return_amount' => $customer['cheque_return_amount'] + $returnReceipt['total_amount'],
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                    }
                }

                $CheqPayCustCheqReceipt = CheqPayCustCheqReceipt::where('cheque_receipt_code', '=', $returnReceipt->code)
                    ->where('is_active', '=', 1)->get();

                foreach ($CheqPayCustCheqReceipt as $Cheq_Pay_Cust_Cheq_Receipt) {

                    $ChequePayment = ChequePayment::where('code', $Cheq_Pay_Cust_Cheq_Receipt['cheque_paym_code'])->first();

                    if ($ChequePayment) {
                        $totalPaidAmount = $ChequePayment['paid_amount'] - $Cheq_Pay_Cust_Cheq_Receipt['paid_amount'];

                        $ChequePayment->update([
                            'paid_amount' => $totalPaidAmount,
                            'status' => ChequeStatusCode::RETURNED_CHEQUE_SETTLED_REVERSED,
                            'paid_status' => PaymentStatus::PAYMENT_NOT_DONE,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow(),
                        ]);
                    }
                    $Cheq_Pay_Cust_Cheq_Receipt->update([
                        'status' => MultiPurposeStatus::REVERSED,

                    ]);
                }

                DB::commit();
                return response()->json([
                    'status' => 200,
                    'returnReceipt' => $returnReceipt,
                    'message' => 'Reversed succesfully'
                ]);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'Return Receipt Code - ' . $CusCheqReturnReceiptCode . ' Not Found'
                ]);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

}
