<?php

namespace App\Http\Controllers\Payment_Controllers;

use App\Enums\ChequeStatusCode;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Enums\TransactionCode;
use App\Enums\MultiPurposeStatus;
use App\Models\Customer_Models\Customer;
use App\Models\PaymentReceipt_Models\ChequePayment;
use App\Models\PaymentReceipt_Models\ModeOfPayment;
use App\Models\PaymentReceipt_Models\PaymentReceipt;
use Exception;
use Illuminate\Support\Facades\DB;

class ChequePaymentController extends Controller
{

        /**
     * Author: Suhail Jamaldeen
     * Date: 11.01.2023
     * Version: 01
     * Logic: Get the check payment details along with other data filter
     * with current logged in location
     *
     * @param  mixed $request
     * @return void
     */
    public function index(Request $request)
    {
        try {
            $currentLocation = getCurrentLocationCode($request);
            $chequePayment = ChequePayment::select(
                'code',
                'stock_location_code',
                'mode_of_payment_code',
                'bank_code',
                'branch_code',
                'number',
                'deposit_date',
                'amount'
            )
            ->with([
                'bank' => function ($query) {
                    $query->where('is_active', '=', 1);

                },
            ])
            ->with([
                    'modeOfPayment' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'modeOfPayment.paymentReceipt' => function ($query) use ($currentLocation) {
                        $query
                        ->where('is_active', '=', 1)
                        ->where('stock_location_code', '=', $currentLocation);
                    },
                    'modeOfPayment.paymentReceipt.customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }, 'modeOfPayment.paymentReceipt.customer.route' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }, 'modeOfPayment.paymentReceipt.customer.route.salesRep' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->orderBy('created_at', 'DESC')
                ->Where('status', '=', $request->status)
                ->get();

            return response()->json([
                'status' => 200,
                'chequePayment' => $chequePayment
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * show
     * Author: Suhail Jamaldeen
     * Date: 11.01.2023
     * Version: 01
     * Logic:
     * @param  mixed $id
     * @return void
     */
    public function show($id)
    {
        try {

            $chequePayment = ChequePayment::select(
                'code',
                'mode_of_payment_code',
                'bank_code',
                'branch_code',
                'number',
                'deposit_date',
                'amount',
                'remarks'
            )->with([
                    'modeOfPayment' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'modeOfPayment.paymentReceipt' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'modeOfPayment.paymentReceipt.customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->Where('code', '=', $id)
                ->get();

            return response()->json([
                'status' => 200,
                'chequePayment' => $chequePayment
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function chequeSearch($key)
    {
        try {

            $chequePayment = ChequePayment::select(
                'cheque_payments.code',
                'cheque_payments.stock_location_code',
                'cheque_payments.mode_of_payment_code',
                'cheque_payments.bank_code',
                'cheque_payments.branch_code',
                'cheque_payments.number',
                'cheque_payments.deposit_date',
                'cheque_payments.amount',
                'cheque_payments.remarks',
                'customers.first_name',
                'users.username'
            )
            ->join(
                'customers',
                'customers.code',
                '=',
                'cheque_payments.customer_code'
            )
            ->join(
                'routes',
                'routes.code',
                '=',
                'customers.route_code'
            )
            ->join(
                'users',
                'users.code',
                '=',
                'routes.rep_code'
            )
            ->join(
                'payment_receipts',
                'payment_receipts.customer_code',
                '=',
                'customers.code'
            )
            ->join(
                'mode_of_payments',
                'mode_of_payments.payment_receipt_code',
                '=',
                'payment_receipts.code'
            )
            ->with([
                'bank' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
            ])
            ->where(function ($query) use ($key) {
                $query->Where('cheque_payments.code', 'like', "%$key%")
                ->orWhere('cheque_payments.deposit_date', 'like', "%$key%")
                ->orWhere('customers.first_name', 'like', "%$key%")
                ->orWhere('cheque_payments.amount', 'like', "%$key%")
                ->orWhere('users.username', 'like', "%$key%");
            })
            ->where('cheque_payments.is_active', '=', 1)
            ->get();


        return response()->json([
            'status' => 200,
            'chequePayment' => $chequePayment
        ]);

        } catch (\Exception $e) {
        throw new Exception($e);
        }
    }

    public function periociChequeSearch(Request $request)
    {
        try {
        $currentLocation = getCurrentLocationCode($request);
        $customer = $request->customer;
        $query = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.created_at',
            'cheque_payments.amount',
            'cheque_payments.status',
            'cheque_payments.date_time'
        )
            ->with([
                'bank' => function ($query) {
                    $query->select('code', 'name')
                        ->where('is_active', '=', 1);
                },
            ])
            ->with([
                'modeOfPayment' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'modeOfPayment.paymentReceipt' => function ($query) use ($currentLocation) {
                    $query
                        ->where('is_active', '=', 1)
                        ->where('stock_location_code', '=', $currentLocation);
                },
                'modeOfPayment.paymentReceipt.customer' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'modeOfPayment.paymentReceipt.customer.route' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'modeOfPayment.paymentReceipt.customer.route.salesRep' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->where('status','=',ChequeStatusCode::CHEQUE_IN_HAND)
            ->whereBetween('cheque_payments.deposit_date', [date('Y-m-d H:m:s', strtotime($request->fromDate)-1), date('Y-m-d H:m:s', strtotime($request->toDate))])
            ->whereHas('modeOfPayment.paymentReceipt.stockLocation', function ($query) use ($request) {
                $query->where('code', '=', getCurrentLocationCode($request));
            })
            ->orderBy('cheque_payments.deposit_date', 'asc');

        // Conditionally apply the customer filter
        if ($customer) {
            $query->whereHas('modeOfPayment.paymentReceipt.customer', function ($query) use ($customer) {
                $query->where('code', '=', $customer);
            });
        }

        $periodicCheque = $query->get();

        return response()->json([
            'status' => 200,
            'chequePayment' => $periodicCheque
        ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    public function store(Request $request)
    {
        try {

            DB::beginTransaction();

            $payment_code = TransactionCode::CHEQUE_PAYMENT_CODE;
            $cus_code = ChequePayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $chequePayment = ChequePayment::create([
                'code' => $payment_code . $max_id,
                'bank_code' => $request->bankCode,
                'branch_code' => $request->branchCode,
                'number' => $request->number,
                'mode_of_payment_code' => $request->modeOfPaymentCode,
                //'customer_code' => $request->customerCode,
                'deposit_date' => $request->date,
                'amount' => $request->amount,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),

            ]);
            DB::commit();

            return response()->json(['status' => 200, 'message' => "Payment  created"]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function edit($id)
    {
        //
    }


    public function update(Request $request, $id)
    {
        try {

            DB::beginTransaction();
            $chequePayment = ChequePayment::where('code', $id)->first();
            $chequePayment->update([
                'bank_code' => $request->bankCode,
                'branch_code' => $request->branchCode,
                'number' => $request->number,
                'mode_of_payment_code' => $request->modeOfPaymentCode,
                'deposit_date' => $request->depositDate,
                'amount' => $request->amount,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
            ]);
            DB::commit();
            return response()->json(['status' => 200, 'message' => "Payment  updated"]);
        } catch (\Exception $e) {

            DB::rollBack();
            throw new Exception($e);
        }
    }


    /**
     * chequeProcess
     *
     * @param  mixed $request
     * @param  mixed $status
     * @return void
     * Author: Suhail Jamaldeen
     * Date: 13.12.2022
     * Version: 01
     * Logic: Change the cheque status and sum the customer checque returned code if the cheque is returned.
     *
     */
    public function chequeProcess(Request $request, $status)
    {
        try {
            $cheques = $request->selected;

            foreach ($cheques as $cheque) {
                $chequePayment = ChequePayment::select(
                    'cheque_payments.code',
                    'cheque_payments.amount',
                    'customers.code AS customerCode',
                    'cheque_payments.status'
                )
                    ->join(
                        'mode_of_payments',
                        'mode_of_payments.code',
                        '=',
                        'cheque_payments.mode_of_payment_code'
                    )
                    ->join(
                        'payment_receipts',
                        'payment_receipts.code',
                        '=',
                        'mode_of_payments.payment_receipt_code'
                    )
                    ->join(
                        'customers',
                        'customers.code',
                        '=',
                        'payment_receipts.customer_code'
                    )
                    ->where('cheque_payments.code', $cheque)
                    ->first();
                ChequePayment::where('code', $cheque)
                    ->first()
                    ->update([
                        'status' => $status,
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()
                    ]);


                if ($status == ChequeStatusCode::CHEQUE_RETURNED) {
                    $customer = Customer::where('code', $chequePayment['customerCode'])
                        ->first();
                    $customer->update([
                        'cheque_return_amount' => $customer['cheque_return_amount'] + ($chequePayment->amount),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()
                    ]);
                }
            }


            return response()->json([
                'status' => 200,
                'message' => "updated"
            ]);

        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    /**
     * thirdPartyCheques
     * autor : fathima sajana
     * version: 1
     * date: 2023.11.27
     * logic:used 3rd party cheques for xpenses and vouchers
     * @param  mixed $request
     * @return void
     */
    public function thirdPartyCheques(Request $request)
    {
        try {
            $thirdPartyCheques = ChequePayment::select(
            'cheque_payments.code',
            'cheque_payments.mode_of_payment_code',
            'cheque_payments.bank_code',
            'cheque_payments.branch_code',
            'cheque_payments.number',
            'cheque_payments.deposit_date',
            'cheque_payments.amount',
            'cheque_payments.remarks',
            'customers.first_name'

        ) ->join(
            'customers',
            'customers.code',
            '=',
            'cheque_payments.customer_code'
        )
        ->where('cheque_payments.status', '=', 13)
        ->get();

        return response()->json([
            'status' => 200,
            'message' => $thirdPartyCheques
        ]);

    } catch (\Exception $e) {
        throw new Exception($e);
    }
    }

    public function chequeCancel($code)
    {
        try {
            $chequePayment = ChequePayment::select('code', 'is_active', 'mode_of_payment_code')
            ->where('code', $code)->where('is_active', '=', 1)->first();
            $chequePayment->update([
                'is_active' => 0
            ]);

            $mode_of_payment = ModeOfPayment::select('code', 'is_active', 'payment_receipt_code')
            ->where('code', '=', $chequePayment->mode_of_payment_code)->where('is_active', '=', 1)->first();
            $mode_of_payment->update([
                'is_active' => 0
            ]);

            $payment_receipt = PaymentReceipt::select('code', 'is_active')
            ->where('code', '=', $mode_of_payment->payment_receipt_code)->where('is_active', '=', 1)->first();
            $payment_receipt->update([
                'is_active' => 0
            ]);

            return response()->json([
                'status' => 200,
                'message' => "Cheque Cancel"
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function destroy($id)
    {
        try {
            $chequePayment = ChequePayment::select('id', 'is_active')->where('code', $id)->where('is_active', '=', 1)->first();
            $chequePayment->update([
                'is_active' => 0
            ]);
            return response()->json([
                'status' => 204,
                'message' => "Payment - Deleted"
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }
}
