<?php

namespace App\Http\Controllers\Invoice_Controllers;

use App\Enums\MultiPurposeStatus;
use App\Enums\PaymentStatus;
use App\Enums\ReturnType;
use App\Enums\StockHistorySource;
use App\Http\Controllers\Function_Controllers\StockFunction;
use App\Models\Stock_Models\Stock;
use Illuminate\Http\Request;
use App\Enums\TransactionCode;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Stock_Models\StockHistory;
use App\Http\Controllers\Function_Controllers\StockHistoryFunction;
use App\Models\Customer_Models\Customer;
use App\Models\Invoice_Models\SalesReturn;
use Exception;

class SalesReturnController extends Controller
{

    private $stock_history_function;
    private $stock_function;

    public function __construct()
    {
        $this->stock_history_function = new StockHistoryFunction();
        $this->stock_function = new StockFunction();

    }

    public function index(Request $request)
    {
        try {
            $salesReturn = SalesReturn::select('code','stock_location_code', 'customer_code','discount_amount', 'invoice_code', 'total_amount', 'remarks', 'paid_status', 'return_date_time', 'status')
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'invoice' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('CODE', 'DESC')
                ->get();
            return response()->json(['status' => 200, 'salesReturn' => $salesReturn]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function show($id)
    {
        try {
            $salesReturn = SalesReturn::select(
                'code',
                'customer_code',
                'invoice_code',
                'total_amount',
                'remarks',
                'return_type',
                'discount_amount',
                'status',
                'status',
                'paid_status',
                'return_date_time',
                'stock_location_code'
            )
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'invoice' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'invoice.stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'invoice.stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                ])
                ->where('is_active', '=', 1)
                ->where('code', $id)
                ->first();
            return response()->json(['status' => 200, 'salesReturn' => $salesReturn]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    public function create()
    {
        //
    }

    public function salesReturnSearchForInsert($key)
    {
        try {
            $invoice = SalesReturn::select(
                'code',
                'customer_code',
                'invoice_code',
                'total_amount',
                'remarks',
                'discount_amount',
                'status',
                'status',
                'paid_status',
                'return_date_time',
                'stock_location_code'
            )

            ->with([
                'customer' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'customer.customerAddresses' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'invoice' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'invoice.stockHistories' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'invoice.stockHistories.itemMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockLocation' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'stockHistories.itemMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
            ])

                ->where([['is_active', '=', 1]])

                ->where(function ($query) use ($key) {
                    $query->Where('code', 'like', "%$key%")
                        ->orWhere('invoice_code', 'like', "%$key%")
                        ->orWhere('customer_code', 'like', "%$key%");
                })
                ->first();
            return $invoice;
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    /**
     * store
     * Author: Suhail Jamaldeen
     * Version: 01
     * Date: 18.12.2022
     * Logic: This will store the sales return and will set the values in stock
     * @param  mixed $request
     * @return void
     */
    public function store(Request $request)
    {

        try {
            $stock_history = $request->returnData;
            DB::beginTransaction();

            if (count($stock_history) > 0) {

                $noteCode = TransactionCode::SALES_RETURN_CODE;
                $stockLocationCode = getCurrentLocationCode($request);
                $locationPrefix = substr("$stockLocationCode", -2);
                $max_code = SalesReturn::select('code')
                    ->where('stock_location_code', '=', $stockLocationCode)
                    ->max('code');

                $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

                $salesReturn = SalesReturn::create([
                    'code' => $noteCode . $locationPrefix . $max_id,
                    'customer_code' => $request['customerCode'] ?? null,
                    'invoice_code' => $request['invoiceCode'] ?? null,
                    'sub_total_amount' => $request['subTotalAmount'] ?? null,
                    'discount_amount' => $request['discountAmount'] ?? null,
                    'total_amount' => $request['totalAmount'] ?? null,
                    'remarks' => $request->remarks,
                    'stock_location_code' => $stockLocationCode,
                    'paid_status' => $request->paidStatus,
                    'return_type' => $request['returnType'],
                    'return_date_time' => getDateTimeNow(),
                    'created_by' => getUserCode(),
                    'created_at' => getDateTimeNow(),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow()
                ]);

                if ($salesReturn->return_type == ReturnType::SALES_RETURN_WITH_BILL) {
                    if (config('setting.is_batch_enabled')) { // if the condition is true this one execute

                        foreach ($stock_history as $stockHistory) {
                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                            $remainingQuantity = ($stock->remaining_quantity + $stockHistory['quantity']);
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $salesReturn->code, StockHistorySource::SALES_RETURN, $stock);
                        }
                    } else { // if the condition is false this one execute (defaulty we give false), it updates the stock information item by item, based on the item code and location.
                        foreach ($stock_history as $stockHistory) {
                            $stock = Stock::where(
                                'item_code',
                                '=',
                                $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                            )->where(
                                    'stock_location_code',
                                    '=',
                                    $stockLocationCode
                                )
                                ->where('is_freeze', '=', 0)
                                ->first();

                            $remainingQuantity = $stockHistory['quantity'] + $stock->remaining_quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $salesReturn->code, StockHistorySource::SALES_RETURN, $stock);
                        }
                    }
                } else if ($salesReturn->return_type == ReturnType::SALES_RETURN_WITHOUT_BILL) {

                    if (config('setting.is_batch_enabled')) { // it updates the stock information in a batch for all items in $stock_history
                        foreach ($stock_history as $stockHistory) {
                            $stock = $this->stock_function->createStock($stockHistory, $salesReturn->code, $stockLocationCode);
                            $remainingQuantity = $stockHistory['quantity'];
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $this->stock_history_function->addStockHistory($stockHistory, $salesReturn->code, StockHistorySource::SALES_RETURN, $stock);
                        }

                    } else { //it updates the stock information item by item, based on the item code and location.

                        foreach ($stock_history as $stockHistory) {
                            $stock = Stock::where(
                                'item_code',
                                '=',
                                $stockHistory['itemCode'] ?? $stockHistory['itemMaster']['code']
                            )->where(
                                    'stock_location_code',
                                    '=',
                                    $stockLocationCode
                                )
                                ->where('is_freeze', '=', 0)
                                ->first();

                            $remainingQuantity = $stockHistory['quantity'] + $stock->remaining_quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                            $stockHistory['totalAmount'] =   $stockHistory['quantity'] *   $stockHistory['retailPrice'];
                            $this->stock_history_function->addStockHistory($stockHistory, $salesReturn->code, StockHistorySource::SALES_RETURN, $stock);
                        }

                    }
                }
                // If the customer is paid back via cash, there is no need to update the return amount in the customer table
                if ($salesReturn->paid_status != PaymentStatus::PAYMENT_GIVING_TO_CUSTOMER) {
                    $customer = Customer::where('code', $salesReturn->customer_code)->first();
                    $customer->update([
                        'return_amount' => $customer['return_amount'] + ($salesReturn->total_amount),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()
                    ]);
                }

                $salesReturnCreated = $this->salesReturnSearchForInsert($salesReturn->code);

                DB::commit();

                return response()->json(
                    [
                        'status' => 200,
                        'salesReturn' => "SalesReturn created",
                        'code' => $salesReturn->code,
                        'submittedData' => $salesReturnCreated,
                    ]
                );
            }
        } catch (Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    /**
     * Author: Fathima Sajana
     * Verion: 01
     * Date: 19.10.2023
     * Logic: Reverse the Sales Return
     *
     * Author: Suhail Jamaldee
     * Version: 02
     * Date: 19.10.2023
     * Logic: Implement Reversal for Batch
     * Summary of reversal
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse|mixed
     */
    public function reversal(Request $request)
    {
        try {
            $salesReturnCode = $request->transactionCode;
            DB::beginTransaction();

            $salesReturn = SalesReturn::where('code', '=', $salesReturnCode)
                ->where('is_active', '=', 1)
                ->first();

            if (!$salesReturn) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Sales Return Code - ' . $salesReturnCode . ' Not Found'
                ]);
            }


            $stockHistories = StockHistory::where('source_code', '=', $salesReturnCode)
                ->where('is_active', '=', 1)
                ->get();

            if ($stockHistories->isNotEmpty()) {

                if (config('setting.is_batch_enabled')) {

                    foreach ($stockHistories as $stockHistory) {

                        $stock = $this->stock_function->getExistingStock($stockHistory->stock_code);
                        if ($stock) {
                            $remainingQuantity = $stock->remaining_quantity - $stockHistory->quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                        }

                        $stockHistory->update([
                            'status' => MultiPurposeStatus::REVERSED,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                    }

                } else {

                    foreach ($stockHistories as $stockHistory) {
                        $itemCode = $stockHistory->itemCode ?: $stockHistory->itemMaster->code;
                        $stock = Stock::where('item_code', '=', $itemCode)->first();

                        if ($stock) {
                            $remainingQuantity = $stock->remaining_quantity - $stockHistory->quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);
                        }
                        $stockHistory->update([
                            'status' => MultiPurposeStatus::REVERSED,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);
                    }

                    $stockHistory->update([
                        'status' => MultiPurposeStatus::REVERSED,
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()
                    ]);
                }


            }

            if ($salesReturn->paid_status !=  PaymentStatus::PAYMENT_GIVING_TO_CUSTOMER) {
                // Reverse the customer return amount
                $customer = Customer::where('code', $salesReturn->customer_code)->first();
                $customer->update([
                    'return_amount' => $customer->return_amount - $salesReturn->total_amount,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow()
                ]);
            }

            // Change the status of the sales return to reverse
            $salesReturn->update([
                'status' => MultiPurposeStatus::REVERSED,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);

            DB::commit();

            return response()->json([
                'status' => 200,
                'message' => $salesReturn
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 500,
                'message' => 'An error occurred: ' . $e->getMessage()
            ]);
        }
    }

    public function edit($id)
    {
        //
    }


    public function update(Request $request, $id)
    {

        DB::beginTransaction();
        try {

            $return_value = $request->returnValue;
            if (count($return_value) > 0) {
                foreach ($return_value as $returnValue) {
                    $salesReturn = SalesReturn::where('is_active', '=', 1)->where('code', $id)->first();
                    $stockLocation = getCurrentLocationCode($request);

                    $salesReturn->update([
                        'customer_code' => $request->customerCode['code'],
                        'invoice_code' => $returnValue['invoiceCode']['code'],
                        'total_amount' => $request->totalAmount,
                        'remarks' => $returnValue['remarks'],
                        'discount_amount' =>$request->discountAmount,
                        'paid_status' => $returnValue['paidStatus'],
                        'return_date_time' => getDateTimeNow(),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()
                    ]);

                    if ($salesReturn) {

                        $stock_history = $request->stockHistory;
                        if (count($stock_history) > 0) {
                            foreach ($stock_history as $stockHistory) {
                                $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                                $remainingQuantity = $stock->remaining_quantity + $stockHistory['$quatity'];
                                $stock->update([
                                    'remaining_quantity' => $remainingQuantity,
                                    'updated_by' => getUserCode(),
                                    'updated_at' => getDateTimeNow()
                                ]);


                                $this->stock_history_function->updateStockHistory($stockHistory, $salesReturn->code, StockHistorySource::SALES_RETURN, $stock);
                            }
                        }
                    }
                }
            }


            DB::commit();


            return response()->json(['status' => 200, 'SalesReturn' => "SalesReturn updated"]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }


    public function destroy($id)
    {
        try {

            $salesReturn = SalesReturn::where('is_active', '=', 1)->where('code', $id)->first();

            $salesReturn->update([
                'is_active' => 0,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
            ]);
            return response()->json(['status' => 200, 'SalesReturn' => "SalesReturn Deleted"]);
        } catch (\Exception $e) {

            throw new Exception($e);
        }
    }
}
