<?php

namespace App\Http\Controllers\Invoice_Controllers;

use App\Enums\DefaultValues;
use App\Enums\StockHistorySource;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Function_Controllers\StockFunction;
use Illuminate\Http\Request;
use App\Models\Invoice_Models\Invoice;
use App\Enums\TransactionCode;
use App\Models\Vehicle_Models\VehicleMaster;
use App\Models\Stock_Models\StockHistory;
use App\Models\Stock_Models\SalesRep;
use App\Models\Stock_Models\Route;
use App\Http\Controllers\Function_Controllers\PaymentReceiptFunction;
use App\Http\Controllers\Function_Controllers\PaymentReceiptFunctionForInvoice;
use App\Http\Controllers\Function_Controllers\StockHistoryFunction;
use App\Models\Customer_Models\Customer;
use App\Models\Invoice_Models\SalesReturn;
use App\Models\Stock_Models\Stock;
use App\Models\Invoice_Models\InvoiceExt;
use Illuminate\Support\Facades\DB;
use Exception;


class InvoiceAutoCareController extends Controller
{
    private $stock_history_function;
    private $stock_function;
    private $payment_receipt_function_for_invoice;

    public function __construct()
    {
        $this->stock_history_function = new StockHistoryFunction();
        $this->payment_receipt_function_for_invoice = new PaymentReceiptFunctionForInvoice();
        $this->stock_function = new StockFunction();
    }


    public function index()
    {

        try {
            $invoices = Invoice::select('code', 'invoice_number', 'invoice_source_code', 'date_time', 'customer_code', 'sales_quotation_code', 'sales_order_code', 'discount_amount', 'discount_percentage', 'discount_total', 'bill_discount', 'paid_amount', 'sub_total_amount', 'total_amount', 'paid_status', 'created_by', 'updated_by')
                ->with([
                    'stockHistories' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.vehicleMasters' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'invoiceExt' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('invoices.is_active', '=', 1)
                ->get();

            // if ($invoices) {
            //    $invoiceExt = InvoiceExt::select('code', 'invoice_code', 'vehicle_master_code', 'odo_reading')

            //     ->with(['invoice.vehicleMaster' => function ($query) {
            //         $query->select('code', 'description')->where('is_active', '=', 1);
            //     }
            //     ])
            //     ->with(['invoice.vehicleMaster.customer'  => function ($query) {
            //         $query->where('is_active', '=', 1);
            //     }
            //     ])
            //     ->with(['invoice.salesOrder' => function ($query) {
            //         $query->where('is_active', '=', 1);
            //     }
            //     ])
            //     ->where('invoice_exts.is_active', '=', 1)
            //     ->get();
            // }

            return response()->json([
                'status' => 200,
                'Invoice' => $invoices,
                //'invoiceExt' => $invoiceExt
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function recentInvoice(Request $request)
    {

        try {
            $key = $request->key;
            $invoices = Invoice::select('invoices.code', 'invoices.invoice_number', 'invoices.date_time', 'customer_code', 'invoices.paid_amount', 'invoices.sub_total_amount', 'remarks', 'invoices.total_amount', 'invoices.paid_status', 'invoices.prints', 'created_by')
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->Where('code', 'like', "%$key%")
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->limit(3)->orderBy('invoices.code', 'DESC')->get();

            return response()->json(['status' => 200, 'Invoice' => $invoices]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function outstandingInvoiceForCustomer($key)
    {
        try {
            $invoices = Invoice::select('invoices.code', 'customer_code', 'customers.first_name', 'total_amount', 'invoices.date_time', 'invoices.total_amount', 'invoices.paid_amount')
                ->join('customers', 'customers.code', '=', 'invoices.customer_code')
                ->where([
                    ['invoices.is_active', '=', 1],
                ])
                ->whereColumn(
                    'invoices.total_amount',
                    '>',
                    'invoices.paid_amount'
                )
                ->where(function ($query) use ($key) {
                    $query->Where('customers.code', 'like', "%$key%");
                })->get();
            return response()->Json(['status' => 200, 'invoices' => $invoices]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }


    public function show($id)
    {
        // try {
        $invoice = Invoice::select(
            'code',
            'invoice_number',
            'invoice_source_code',
            'date_time',
            'customer_code',
            'sales_quotation_code',
            'sales_order_code',
            'discount_amount',
            'discount_percentage',
            'bill_discount',
            'discount_total',
            'paid_amount',
            'sub_total_amount',
            'total_amount',
            'paid_status',
            'prints',
            'received_amount',
            'created_by'
        )
            ->with([
                'stockHistories' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'stockHistories.itemMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'stockHistories.itemMaster.itemUnit' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
            ])
            ->with([
                'customer' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'customer.route' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'customer.route.salesRep' => function ($query) {
                    $query->where('is_active', '=', 1);
                },
                'customer.customerAddresses' => function ($query) {
                    $query->where('is_active', '=', 1);
                },

            ])
            ->with([
                'invoiceExt' => function ($query) {
                    $query->select(
                        'code',
                        'invoice_code',
                        'vehicle_master_code',
                        'odo_reading',
                        'next_service_km',
                        'next_service_date'
                    );
                },
                'invoiceExt.vehicleMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'salesReturns' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'salesQuotation' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->where('code', $id)
            ->where('is_active', '=', 1)
            ->first();


        //$invoiceExt  = InvoiceExt::select('code', 'invoice_code', 'vehicle_master_code', 'odo_reading')
        // ->join('vehicle_masters', 'vehicle_masters.code', 'invoice_exts.vehicle_master_code')
        // // ->with(['invoice' => function ($query) {
        // //     $query->where('is_active', '=', 1);
        // // }
        // // ])
        // ->with(['invoice.vehicleMaster' => function ($query) {
        //     $query->select('code', 'description')->where('is_active', '=', 1);
        // }
        // ])
        // ->with(['invoice.vehicleMaster.customer'  => function ($query) {
        //     $query->where('is_active', '=', 1);
        // }
        // ])
        //->where('invoice_code', $invoice->code)
        // ->where('is_active', '=', 1)
        // ->first();


        $salesReturn = SalesReturn::select(
            'code',
            'invoice_code',
            'total_amount',
            'remarks',
            'customer_code',
            'total_amount',
            'return_date_time'
        )
            ->with([
                'stockHistories' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'stockHistories.itemMaster' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'customer' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->with([
                'customer.customerAddresses' => function ($query) {
                    $query->where('is_active', '=', 1);
                }
            ])
            ->where('is_active', '=', 1)
            ->where('invoice_code', $id)
            ->get();



        return response()->json(
            [
                'status' => 200,
                'invoice' => $invoice,
                //'invoiceExt' => $invoiceExt,
                'salesReturn' => $salesReturn
            ]
        );
        // } catch (\Exception $e) {
        //     throw new Exception($e);
        // }
    }
    public function itemPriceSearchLocationBased(Request $request)
    {
        $key = $request->key;
        try {
            $stock = Stock::select(
                'code',
                'code as stockCode',
                'remaining_quantity',
                'wholesale_price',
                'retail_price',
                'cost_price',
                'stock_location_code',
                'item_code'
            )->where([['remaining_quantity', '>', 0]]) //Comment if you don't want to display 0 remaining quantity
                ->where('stock_location_code', '=', $request->header('LocationCode'))
                ->with([
                    'itemMaster' => function ($query) {
                        $query->select(
                            'code',
                            'name',
                            'barcode',
                            'description',
                            'unit_code'
                        )
                            ->with([
                                'stocks' => function ($query) {
                                            $query->select(
                                                'code',
                                                'item_code',
                                                'code as stockCode',
                                                'remaining_quantity',
                                                'wholesale_price',
                                                'retail_price',
                                                'cost_price',
                                                'stock_location_code',
                                            );
                                        }
                                ,
                                'itemUnit' => function ($query) {
                                            $query->select(
                                                'code',
                                                'name'
                                            )->where('is_active', '=', 1);
                                        }
                            ])
                            ->where('is_active', '=', 1);
                    }
                ])
                ->whereHas('itemMaster', function ($query) use ($key) {
                    $query->where('name', 'LIKE', '%' . $key . '%')
                        ->orwhere('barcode', 'LIKE', '%' . $key . '%');

                })
                // ->orwhereHas('itemMaster', function ($query) use ($key) {
                //     $query->where('barcode', 'LIKE', '%' . $key . '%');
                // })

                ->get();

            return response()->json([
                'status' => 200,
                'stock' => $stock

            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function invoiceSearch($key)
    {
        try {
            $invoice = Invoice::select(
                'code',
                'invoice_number',
                'invoice_source_code',
                'date_time',
                'customer_code',
                'sales_quotation_code',
                'sales_order_code',
                'discount_amount',
                'discount_percentage',
                'bill_discount',
                'discount_total',
                'paid_amount',
                'sub_total_amount',
                'total_amount',
                'prints',
                'paid_status',
                'received_amount',
                'created_by',
                'updated_by'
            )

                ->with([
                    'stockHistories.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->with([
                    'customer.route.salesRep' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                // ->with(['invoicePayment' => function ($query) {
                //     $query->where('is_active', '=', 1);
                // }])
                // ->with(['salesOrders' => function ($query) {
                //     $query->where('is_active', '=', 1);
                // }])
                ->where([['is_active', '=', 1]])
                //->leftjoin('sales_reps', 'sales_reps.code', '=', 'customers.rep_code')

                ->where(function ($query) use ($key) {
                    $query->Where('code', 'like', "%$key%")
                        ->orWhere('invoice_number', 'like', "%$key%")
                        ->orWhere('customer_code', 'like', "%$key%");
                })
                ->get();

            if ($invoice) {
                $invoiceExt = InvoiceExt::select('code', 'invoice_code', 'vehicle_master_code', 'odo_reading')

                    ->with([
                        'invoice.vehicleMaster' => function ($query) {
                            $query->where('is_active', '=', 1);
                        }
                    ])
                    ->with([
                        'invoice.vehicleMaster.customer' => function ($query) {
                            $query->where('is_active', '=', 1);
                        }
                    ])
                    ->where('is_active', '=', 1)
                    ->get();
            }
            return response()->json([
                'status' => 200,
                'invoice' => $invoice,
                'invoiceExt' => $invoiceExt
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }



    public function create()
    {
        //
    }

    /**
     * Author: Fathima Sajana
     * Date: 15.02.2023
     * Logic: Stores the Invoice Details in invoice and invoice ext tables, stock History, mode of payment, update vechicle master and
     * customer
     * @param  mixed $request
     * @return void
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $invoice_code = TransactionCode::INVOICE_CODE;

            $stockLocationCode = getCurrentLocationCode($request);
            $locationPrefix = substr($stockLocationCode, -2);
            $max_code = Invoice::select('code')
                ->where('stock_location_code', '=', $stockLocationCode)
                ->max('code');
            $max_id = $max_code == null ? config('global.code_value') + 1 : substr($max_code, 5) + 1;

            $invoice = Invoice::create([
                'code' => $invoice_code . $locationPrefix . $max_id,
                'invoice_number' => $request->invoiceNumber,
                'stock_location_code' => getCurrentLocationCode($request),
                'invoice_source_code' => $request->invoiceSourceCode,
                'date_time' => getDateTimeNow(),
                'customer_code' => $request->customer['code'] ?? DefaultValues::CUSTOMER_CODE,
                'sales_quotation_code' => $request->salesQuotation ? $request->salesQuotation['code'] : null,
                'sales_order_code' => $request->salesOrderCode ?? null,
                'discount_amount' => $request->discountAmount,
                'discount_percentage' => $request->discountPercentage,
                'bill_discount' => $request->billDiscount,
                'discount_total' => $request->discountTotal,
                'paid_amount' => $request->paidAmount,
                'sub_total_amount' => $request->subTotalAmount,
                'total_amount' => $request->totalAmount,
                'paid_status' => $request->paidStatus,
                'prints' => 1,
                'is_locked' => $request->isLocked,
                'remarks' => $request->remarks,
                'received_amount' => $request->receivedAmount,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);

            if ($invoice) {


                $this->addInvoiceExt($request, $invoice);

                $stock_history = $request->stockHistories;
                if (count(($stock_history)) > 0) {
                    foreach ($stock_history as $stockHistory) {
                        $quantity = $stockHistory['quantity'];
                        $stockHistory['subTotalAmount'] = $quantity * $stockHistory['retailPrice'];
                        $stockHistory['totalAmount'] = $stockHistory['retailPrice'] ? ($quantity * $stockHistory['retailPrice']) - (($quantity * $stockHistory['retailPrice']) * $stockHistory['discountPercentage'] / 100) - ($quantity * $stockHistory['discountAmount']) : 0;
                        $stockHistory['discountTotal'] = $stockHistory['retailPrice'] ? ($quantity * $stockHistory['discountAmount']) + (($quantity * $stockHistory['retailPrice']) * $stockHistory['discountPercentage'] / 100) : 0;
                        $stockHistory['subTotalAmount'] = $quantity * $stockHistory['retailPrice'];
                        //Unit Sold Price is the amount per item after applying the discount
                        $stockHistory['unitSoldPrice'] = $stockHistory['totalAmount'] / $quantity;

                        if ($stockHistory['isService'] ?? false) {
                            $this->stock_history_function->addStockHistoryForService($stockHistory, $invoice->code, StockHistorySource::INVOICE);
                        } else {

                            $stock = $this->stock_function->getExistingStock($stockHistory['stockCode']);
                            $remainingQuantity = $stock->remaining_quantity - $quantity;
                            $stock->update([
                                'remaining_quantity' => $remainingQuantity,
                                'updated_by' => getUserCode(),
                                'updated_at' => getDateTimeNow()
                            ]);

                            $stockHistoryAdd = $this->stock_history_function->addStockHistory($stockHistory, $invoice->code, StockHistorySource::INVOICE, $stock);

                        }

                    }
                }

                $modeOfPayments = $request->modeOfPayments;
                if ($request->paidAmount != 0) {
                    $this->payment_receipt_function_for_invoice->addPaymentReceipt($request, $modeOfPayments, $invoice);
                }

                // if ($invoice->customer_code != DefaultValues::CUSTOMER_CODE) {
                    $customer = Customer::where('code', $invoice->customer_code)->first();
                    $customer->update([
                        'invoice_total' => $invoice->total_amount + $customer['invoice_total'],
                        'received_amount' => $invoice['paid_amount'] + $customer['received_amount'],
                        'credit_amount' => $invoice['paid_amount'] == 0 ? ($customer['credit_amount'] + $invoice['total_amount']) : $customer['credit_amount'] + ($invoice['total_amount'] - $invoice['paid_amount']),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow(),
                    ]);
                // }

            }

            DB::commit();
            return response()->json([
                'status' => 200,
                'Invoice' => "Invoice created",
                'code' => $invoice->code,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }

    }

    /**
     * addInvoiceExt
     * Author: Suhail Jamaldeen
     * Date: 21.03.2023
     * Logic: Insert Into Invoice Ext
     * Version: 01
     * @param  mixed $request
     * @param  mixed $invoice
     * @return void
     */
    public function addInvoiceExt(Request $request, $invoice)
    {
        try {
            DB::beginTransaction();
            $invoice_code = TransactionCode::INVOICE_EXT_CODE;
            $stockLocationCode = getCurrentLocationCode($request);
            $locationPrefix = substr($stockLocationCode, -2);
            $max_code = Invoice::select('code')
                ->where('stock_location_code', '=', $stockLocationCode)
                ->max('code');
            $max_id = $max_code == null ? config('global.code_value') + 1 : substr($max_code, 5) + 1;

            $vehicle = $request['invoiceExt']['vehicleMaster'] ?? null;
            $odoReading = $request['invoiceExt']['odoReading'] ?? 0;

            $nextServiceKm = $odoReading + ($vehicle ? $vehicle['serviceKm'] : 0);
            $nextServiceDate = date('Y-m-d H:m:s', strtotime("+" . ($vehicle ? $vehicle['servicePeriod'] : 0) . " days"));

            $stock_history_ext = InvoiceExt::create([
                'code' => $invoice_code . $locationPrefix . $max_id,
                'invoice_code' => $invoice->code,
                'vehicle_master_code' => ($vehicle !== null) ? $vehicle['code'] : null,
                'odo_reading' => $odoReading,
                'next_service_km' => $nextServiceKm,
                'next_service_date' => $nextServiceDate,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
            return $stock_history_ext;
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function edit($id)
    {
        //
    }

    /**
     * update
     * Author: Suhail Jamaldeen
     * Date: 20.03.2023
     * Logic: Update the Invoice Details. When editing Vehicle cannot be changed.
     * Version: 01
     * @param  mixed $request
     * @param  mixed $id
     * @return void
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            $invoice = Invoice::where('code', $id)->first();
            // This is used when the vehicle is changed.
            if ($invoice) {
                $customer = Customer::where('code', $invoice->customer_code)->first();
                $customer->update([
                    'invoice_total' => (($customer['invoice_total'] - $invoice['total_amount'])),

                    // when newly start this function for changing customer (already existing invoice customer) , call this one for reducing credit balance from that custoemr
                    'credit_amount' => $customer['credit_amount'] - (($invoice->total_amount) - ($invoice->paid_amount)),
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);
            }

            $invoice->update([
                'invoice_number' => $request->invoiceNumber,
                'invoice_source_code' => $request->invoiceSourceCode,
                'customer_code' => $request->customer['code'] ?? DefaultValues::CUSTOMER_CODE,
                'sales_quotation_code' => $request->salesQuotationCode ?? null,
                'sales_order_code' => $request->salesOrderCode ?? null,
                'discount_amount' => $request->discountAmount,
                'discount_percentage' => $request->discountPercentage,
                'bill_discount' => $request->billDiscount,
                'discount_total' => $request->discountTotal,
                'paid_amount' => $request->paidAmount,
                'sub_total_amount' => $request->subTotalAmount,
                'total_amount' => $request->totalAmount,
                'paid_status' => $request->paidStatus,
                'prints' => $request->prints,
                'is_locked' => $request->isLocked,
                'remarks' => null,
                'received_amount' => $request->receivedAmount,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
            ]);

            if ($invoice) {

                $invoiceExt = $request->invoiceExt;
                $vehicle = $invoiceExt['vehicleMaster'];
                $odoReading = $invoiceExt['odoReading'];
                $nextServiceDate = date('Y-m-d H:m:s', strtotime("+" . ($vehicle ? $vehicle['servicePeriod'] : 0) . " days"));
                $nextServiceKm = $odoReading + ($vehicle ? $vehicle['serviceKm'] : 0);

                // if they changed vehicle master and odo reading , invoice Ext wil be updated
                $invoiceExtChange = InvoiceExt::where('code', $invoiceExt['code'])->first();
                $invoiceExtChange->update([
                    // 'vehicle_master_code' => $vehicle['code'], //Vehicle Cannot be changed.
                    'odo_reading' => $odoReading,
                    'next_service_km' => $nextServiceKm,
                    'next_service_date' => $nextServiceDate,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

                $stockHistories = $request->stockHistories;

                foreach ($stockHistories as $stockHistory) {
                    $stockHistory['totalAmount'] = $stockHistory['quantity'] > 0 ? $stockHistory['retailPrice'] ? ($stockHistory['quantity'] * $stockHistory['retailPrice']) - (($stockHistory['quantity'] * $stockHistory['retailPrice']) * $stockHistory['discountPercentage'] / 100) - ($stockHistory['quantity'] * $stockHistory['discountAmount']) : 0 : 0;
                    $stockHistory['discountTotal'] = $stockHistory['quantity'] > 0 ? $stockHistory['retailPrice'] ? ($stockHistory['quantity'] * $stockHistory['discountAmount']) + (($stockHistory['quantity'] * $stockHistory['retailPrice']) * $stockHistory['discountPercentage'] / 100) : 0 : 0;

                    //Unit Sold Price is the amount per item after applying the discount
                    $stockHistory['unitSoldPrice'] = $stockHistory['quantity'] > 0 ? $stockHistory['totalAmount'] / $stockHistory['quantity'] : 0;


                    if ($stockHistory['itemMaster'] ?? false) {

                        if ($stockHistory['itemMaster']['isService']) {

                            $this->stock_history_function->updateStockHistoryForService($stockHistory, $invoice->code, StockHistorySource::INVOICE);
                        } else {
                            $stock = $this->stock_history_function->stockResetInvoice($request, $stockHistory, StockHistorySource::INVOICE, $invoice);
                            $stock = Stock::select('code', 'remaining_quantity')
                                ->where('code', $stockHistory['stockCode'])
                                ->first();
                            StockHistory::where('code', $stockHistory['stockCode'])
                                ->update([
                                    'remaining_quantity' => $stock['remaining_quantity'] ?? null,
                                ]);

                            $this->stock_history_function->updateStockHistory($stockHistory, $invoice->code, StockHistorySource::INVOICE, $stock);
                        }
                    } else {
                        $this->stock_history_function->addStockHistoryForService($stockHistory, $invoice->code, StockHistorySource::INVOICE);
                    }


                }


                $modeOfPayments = $request->modeOfPayments;

                if ($request->paidAmount != 0) {
                    $this->payment_receipt_function_for_invoice->addPaymentReceipt($request, $modeOfPayments, $invoice);
                }

                if ($invoice->customer_code != DefaultValues::CUSTOMER_CODE) {
                    $customer = Customer::where('code', $invoice->customer_code)->first();
                    $customer->update([
                        // calling this credit balance for newly changed customer (from existing customer to updated customer), this customer alredy had credit balance adding new amount through this credit_amount , if it is not call other fuction ($invoice->total_amount - $invoice->paid_amount)
                        //'credit_amount' => ($invoice->total_amount - $invoice->paid_amount) > 0 ? $customer['credit_amount'] + ($invoice->total_amount - $invoice->paid_amount) : $customer['credit_amount'],
                        'credit_amount' => $customer['credit_amount'] ? ($customer['credit_amount'] + ($invoice->total_amount - $invoice->paid_amount)) : ($invoice->total_amount - $invoice->paid_amount),
                        'invoice_total' => (($customer['invoice_total'] + $invoice['total_amount'])),
                        //'return_amount' => (($customer['return_amount'] + $invoice['total_amount'])),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow(),
                    ]);
                }

            }



            DB::commit();


            return response()->json([
                'status' => 200,
                'Invoice' => "Invoice Updated",
                'code' => $invoice->code,
                'date' => $invoice->updated_at,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function updateInvoiceExt($invoiceCode, $invoiceExt, $vehicleMasterCode, $odoReading)
    {
        try {
            DB::beginTransaction();
            $invoice_ext = InvoiceExt::where('code', $invoiceExt['code'])->where('is_active', '=', 1)->first();
            if ($invoice_ext) {
                $invoice_ext->update([
                    'vehicle_master_code' => $vehicleMasterCode,
                    'odo_reading' => $odoReading,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);
            }

            DB::commit();
            return $invoice_ext;
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function destroy($id)
    {
        try {
            $invoice = Invoice::select('id', 'is_active')->where('code', $id)->where('is_active', '=', 1)->first();
            $invoice->update([
                'is_active' => 0
            ]);
            return response()->json([
                'status' => 204,
                'message' => "Invoice- Deleted"
            ]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }
}
