<?php

namespace App\Http\Controllers\Expense_Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Enums\TransactionCode;
use App\Models\Expense_Models\ExpenseType;
use App\Models\Expense_Models\ExpenseCategory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ExpenseTypeController extends Controller
{
    public function index()
    {
        try {
            $expenseType  = ExpenseType::select('code', 'expense_category_code', 'name', 'description')->where('is_active', '=', 1)->get();
            return response()->json(['status' => 200, 'expenses_types' =>  $expenseType]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $expenseType  = ExpenseType::select('code', 'expense_category_code', 'name', 'description')
                ->with(['expenseCategories' => function ($query) {
                    $query->select('code','name','expense_type_code' )->where('is_active', '=', 1);
                }])
                ->where('code', $id)
                ->where('is_active', '=', 1)
                ->first();
            return response()->json(['status' => 200, 'expenses_types' =>  $expenseType]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }


    public function create()
    {
        //
    }


    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|unique:expense_types',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 401,
                    'errors' => $validator->errors(),
                ], 422);
            }

           
            DB::beginTransaction();
            $expense_code = TransactionCode::EXPENSES_TYPE_CODE;
            $cus_code = ExpenseType::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;


            $expenseType = ExpenseType::create([

                'code' => $expense_code.$max_id,
                'name' => $request->name,
                //'description' => $request->description,
                'expense_category_code'=>$request->expense_category_code,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()

            ]);
            if ($expenseType) {
                $expenseCategory = $request->expenseCategories;
                $this->addExpenseCategory($expenseCategory, $expenseType->code);
            }

            DB::commit();
            return response()->json([
                'status' => 200,
                //'message' => "Expense Types created"
                'message' => $expenseCategory
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    public function edit($id)
    {
    }

    public function update(Request $request, $id)
    {
        try {
           
            DB::beginTransaction();
            $expenseType = ExpenseType::where('code', $id)->first();
            $expenseType->update([
                'name' => $request->name,
                //'description' => $request->description,
                'expense_category_code'=>$request->expense_category_code,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
            ]);
            if ($expenseType) {
                $expenseCategory = $request->expenseCategories;
                $this->updateExpenseCategory($expenseCategory, $expenseType->code);
            }

            DB::commit();
            return response()->json(['status' => 200, 'message' => "Expense Types updated"]);

        } catch (\Exception $e) {
           DB::rollBack();

            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $expenseType = ExpenseType::select('id', 'is_active')->where('code', $id)->where('is_active', '=', 1)->first();
            $expenseType->update([
                'is_active' => 0
            ]);
            return response()->json([
                'status' => 204,
                'message' => "Expense Types - Deleted"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    protected function addExpenseCategory($expenseCategory, $typeCode)
    {
         try {
        if (count($expenseCategory) > 0) {
            foreach ($expenseCategory as $category) {
                $this->createExpenseCategory($category, $typeCode);
            }
        }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    protected function updateExpenseCategory($expenseCategory, $typeCode)
    {
        try {
            
            DB::beginTransaction();
            foreach ($expenseCategory as $category) {
                $expenseCategory = "";
                if (isset($category['code'])) {

                    $expenseCategory = ExpenseCategory::select('id', 'code', 'Expense_type_code', 'name')
                        ->where('code', $category['code'])
                        ->where('Expense_type_code', $typeCode)
                        ->first();
                    $expenseCategory->update([
                        'name' => $category['name'],
                        //'Expense_type_code' => $typeCode,
                        //'description' => $category['description'],
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()

                    ]);
                } else {
                    $this->createExpenseCategory($category, $typeCode);
                }
            }
            DB::commit();
            return response()->json(['status' => 200, 'message' => "Expense Types updated"]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    protected function createExpenseCategory($category, $typeCode)
    {
        try {
           
            DB::beginTransaction();

            $expense_code = TransactionCode::EXPENSES_CATEGORY;
            $cus_code = ExpenseCategory::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;


            ExpenseCategory::create([
                'code' => $expense_code . $max_id,
                'Expense_type_code' => $typeCode,
                'name' => $category['name'],
                //'description' => $category['description'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
            ]);
            DB::commit();
            return response()->json(['status' => 200, 'message' => "Expense Types created"]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }
}
