<?php

namespace App\Http\Controllers\Expense_Controllers;

use App\Enums\ChequeStatusCode;
use App\Enums\LedgerType;
use App\Enums\MultiPurposeStatus;
use App\Enums\PaymentTypeCode;
use Illuminate\Http\Request;
use App\Enums\TransactionCode;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Expense_Models\Expense;
use App\Models\Expense_Models\ExpenseBankPayment;
use App\Models\Expense_Models\ExpenseCardPayment;
use App\Models\Expense_Models\ExpenseCashPayment;
use App\Models\Expense_Models\ExpenseChequePayment;
use App\Models\Expense_Models\ExpenseModeOfPayment;
use App\Models\Expense_Models\ExpenseOnlinePayment;
use Exception;

class ExpenseController extends Controller
{

    public function index(Request $request)
    {
        try {
            $expenses = Expense::select('code', 'expense_type_code', 'expense_category_code','date_time', 'total_amount', 'status', 'created_by', 'description')
                ->with([
                    'expenseType' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    },
                    'expenseCategory' => function ($query) {
                        $query->select('code', 'name')->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'expenseModeOfPayments' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'user' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                // ->join('expense_categories', 'expense_categories.code', '=', 'expenses.expense_category_code')
                // ->join('expense_types', 'expense_types.code', '=', 'expenses.expense_type_code')
                ->where('expenses.is_active', '=', 1)
                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->orderBy('created_at', 'DESC')
                ->get();
            return response()->json(['status' => 200, 'Expenses' => $expenses]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }


    public function show($id)
    {
        try {
            $expense = Expense::select(
                'code',
                'expense_type_code',
                'expense_category_code',
                'date_time',
                'total_amount',
                'status',
                'description',
                'created_by'
            )->with([
                        'expenseType' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                        },
                        'expenseCategory' => function ($query) {
                            $query->select('code', 'name')->where('is_active', '=', 1);
                            // }, 'expensePayment' => function ($query) {
                            //     $query->select('code', 'expense_code', 'payment_type_code', 'bank_code', 'total_amount', 'cheque_date', 'cheque_number')->where('is_active', '=', 1);
                        }
                    ])
                ->with([
                    'expenseModeOfPayments' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'user' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('code', $id)
                ->where('is_active', '=', 1)
                ->first();

            if ($expense) {
                $cash = ExpenseCashPayment::select('amount')
                    ->where('expense_code', $expense->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $cheque = ExpenseChequePayment::select('number', 'bank_code', 'branch_code', 'deposit_date', 'amount')
                    ->with([
                        'bank' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->where('expense_code', $expense->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $bank = ExpenseBankPayment::select('bank_code', 'reference_number', 'date', 'amount')
                    ->with([
                        'bank' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->where('expense_code', $expense->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $card = ExpenseCardPayment::select('bank_code', 'card_number', 'amount')
                    ->with([
                        'bank' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->where('expense_code', $expense->code)
                    ->where('is_active', '=', 1)
                    ->get();

                $onlineTransfer = ExpenseOnlinePayment::select('bank_code', 'transfer_id', 'transfer_date', 'amount')
                    ->with([
                        'bank' => function ($query) {
                            $query->select('code', 'name')
                                ->where('is_active', '=', 1);
                        }
                    ])
                    ->where('expense_code', $expense->code)
                    ->where('is_active', '=', 1)
                    ->get();
            }
            return response()->json([
                'status' => 200,
                'expense' => $expense,
                'cashPayment' => $cash,
                'chequePayment' => $cheque,
                'bankPayment' => $bank,
                'cardPayment' => $card,
                'onlineTransferPayment' => $onlineTransfer
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }


    public function store(Request $request)
    {
        // try {
        DB::beginTransaction();

        $expense_code = TransactionCode::EXPENSES;
        $stockLocationCode = getCurrentLocationCode($request);
        //$ledgerTypeCode = LedgerType::CASH;
        $locationPrefix = substr("$stockLocationCode", -2);
        $max_code = Expense::select('code')
            ->where('stock_location_code', '=', $stockLocationCode)
            ->max('code');

        $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

        $expense = Expense::create([
            'code' => $expense_code . $locationPrefix . $max_id,
            'stock_location_code' => getCurrentLocationCode($request),
            'expense_type_code' => $request->expenseType['code'] ?? null,
            'expense_category_code' => $request->expenseCategory['code'],
            //'ledger_type_code'=> $request->ledgerType['code'],
            'total_amount' => $request->totalAmount,
            'description' => $request->description,
            // 'payment_type_code' => $request->paymentTypeCode,
            // 'bank_code' => $request->bankCode,
            // 'deposite_date' => $request->depositeDate,
            // 'cheque_date' => $request->chequeDate,
            // 'cheque_number' => $request->chequeNumber,
            // 'ref_number' => $request->refNumber,
            'date_time' => $request->dateTime ,
            'status' => MultiPurposeStatus::ACTIVE,
            'created_by' => getUserCode(),
            'created_at' => getDateTimeNow(),
            'updated_by' => getUserCode(),
            'updated_at' => getDateTimeNow()

        ]);
        $expenseModeOfPayments = $request->expenseModeOfPayments;
        if ($expense) {
            if (count($expenseModeOfPayments) > 0) {
                foreach ($expenseModeOfPayments as $expense_mode_of_payment) {
                    $expense_mode_of_payment_code = TransactionCode::EXPENSE_MODE_OF_PAYMENT_CODE;
                    $max_code = ExpenseModeOfPayment::max('code');
                    $max_id = $max_code == null ? config('global.code_value') + 1 : substr("$max_code", 3) + 1;

                    $expenseModeOfPayment = ExpenseModeOfPayment::create([
                        'code' => $expense_mode_of_payment_code . $max_id,
                        'expense_code' => $expense->code,
                        'payment_type_code' => $expense_mode_of_payment['paymentType'],
                        'amount' => $expense_mode_of_payment['amount'],
                        'created_by' => getUserCode(),
                        'created_at' => getDateTimeNow(),
                        'updated_by' => getUserCode(),
                        'updated_at' => getDateTimeNow()
                    ]);

                    if ($expense_mode_of_payment) {
                        if ($expenseModeOfPayment->payment_type_code == PaymentTypeCode::CASH_PAYMENT_CODE) {
                            $this->addCashPayment($expense_mode_of_payment, $expenseModeOfPayment->code, $expense);
                        } else if ($expenseModeOfPayment->payment_type_code == PaymentTypeCode::CHEQUE_PAYMENT_CODE) {
                            $this->addChequePayment($expense_mode_of_payment, $expenseModeOfPayment->code, $expense);
                        } else if ($expenseModeOfPayment->payment_type_code == PaymentTypeCode::BANK_PAYMENT_CODE) {
                            $this->addBankPayment($expense_mode_of_payment, $expenseModeOfPayment->code, $expense);
                        } else if ($expenseModeOfPayment->payment_type_code == PaymentTypeCode::CARD_PAYMENT_CODE) {
                            $this->addCardPayment($expense_mode_of_payment, $expenseModeOfPayment->code, $expense);
                        } else if ($expenseModeOfPayment->payment_type_code == PaymentTypeCode::ONLINE_TRANSFER_PAYMENT_CODE) {
                            $this->addOnlineTransferPayment($expense_mode_of_payment, $expenseModeOfPayment->code, $expense);
                        }
                    }

                }
            }
        }

        DB::commit();
        return response()->json(['status' => 200, 'message' => "Expenses created"]);
        // } catch (\Exception $e) {
        //     DB::rollBack();
        //     return response()->json([
        //         'status' => 500,
        //         'message' => $e
        //     ], 500);
        // }
    }


    public function addCashPayment($expenseModeOfPayment, $expenseModeOfPaymentCode, $expense)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::EXPENSE_CASH_PAYMENT_CODE;
            $cus_code = ExpenseCashPayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $cashPayment = ExpenseCashPayment::create([
                'code' => $payment_code . $max_id,
                'expense_code' => $expense['code'],
                'mode_of_payment_code' => $expenseModeOfPaymentCode,
                'amount' => $expenseModeOfPayment['amount'],
                'date_time' => $expense['date_time'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),

            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function addChequePayment($expenseModeOfPayment, $expenseModeOfPaymentCode, $expense)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::EXPENSE_CHEQUE_PAYMENT_CODE;
            $cus_code = ExpenseChequePayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;


            $cheque_Payment = ExpenseChequePayment::create([
                'code' => $payment_code . $max_id,
                'expense_code' => $expense['code'],
                'mode_of_payment_code' => $expenseModeOfPaymentCode,
                'bank_code' => $expenseModeOfPayment['chequePayment']['bank'],
                'branch_code' => $expenseModeOfPayment['chequePayment']['branchCode'],
                'number' => $expenseModeOfPayment['chequePayment']['chequeNumber'],
                'deposit_date' => $expenseModeOfPayment['chequePayment']['chequeDate'],
                'amount' => $expenseModeOfPayment['amount'],
                'date_time' => $expense['date_time'],
                'remarks' => $expenseModeOfPayment['chequePayment']['remarks'] ?? null,
                'status' => ChequeStatusCode::CHEQUE_IN_HAND,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }


    public function addBankPayment($expenseModeOfPayment, $expenseModeOfPaymentCode, $expense)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::EXPENSE_BANK_PAYMENT_CODE;
            $cus_code = ExpenseBankPayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $bankPayment = ExpenseBankPayment::create([
                'code' => $payment_code . $max_id,
                'expense_code' => $expense['code'],
                'mode_of_payment_code' => $expenseModeOfPaymentCode,
                'bank_code' => $expenseModeOfPayment['bankDeposit']['bank'],
                //$modeOfPayment['bank']['code'],
                'reference_number' => $expenseModeOfPayment['bankDeposit']['refNumber'],
                'amount' => $expenseModeOfPayment['amount'],
                'date_time' => $expense['date_time'],
                'date' => $expenseModeOfPayment['bankDeposit']['date'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
            return response()->Json(['status' => 200, 'message' => 'Bank Payment Updated']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function addCardPayment($expenseModeOfPayment, $expenseModeOfPaymentCode, $expense)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::EXPENSE_CARD_PAYMENT_CODE;
            $cus_code = ExpenseCardPayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $CardPayment = ExpenseCardPayment::create([
                'code' => $payment_code . $max_id,
                'expense_code' => $expense['code'],
                'mode_of_payment_code' => $expenseModeOfPaymentCode,
                'bank_code' => $expenseModeOfPayment['cardPayment']['bank'],
                'card_number' => $expenseModeOfPayment['cardPayment']['cardNumber'],
                'amount' => $expenseModeOfPayment['amount'],
                'date_time' => $expense['date_time'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),

            ]);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function addOnlineTransferPayment($expenseModeOfPayment, $expenseModeOfPaymentCode, $expense)
    {
        try {
            DB::beginTransaction();

            $payment_code = TransactionCode::EXPENSE_ONLINE_TRANSFER_PAYMENT_CODE;
            $cus_code = ExpenseOnlinePayment::max('code');
            $max_id = $cus_code == null ? config('global.code_value') + 1 : substr("$cus_code", 3) + 1;

            $onlineTransferPayment = ExpenseOnlinePayment::create([
                'code' => $payment_code . $max_id,
                'expense_code' => $expense['code'],
                'mode_of_payment_code' => $expenseModeOfPaymentCode,
                'bank_code' => $expenseModeOfPayment['onlineTransfer']['bank'],
                //$modeOfPayment['bank']['code'],
                'transfer_id' => $expenseModeOfPayment['onlineTransfer']['transferId'],
                'amount' => $expenseModeOfPayment['amount'],
                'date_time' => $expense['date_time'],
                'remarks' => $expenseModeOfPayment['remarks'] ?? null,
                'transfer_date' => $expenseModeOfPayment['onlineTransfer']['transferDate'] ?? null,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),
            ]);
            DB::commit();
            return response()->Json(['status' => 200, 'message' => 'Online transfer Payment Updated']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    public function edit($id)
    {
        //
    }

    public function reversal(Request $request)
    {
        try {
            $expenseCode = $request->transactionCode;

            $expense = Expense::where('code', '=', $expenseCode)
                ->where('status', '<>', MultiPurposeStatus::REVERSED)
                ->where('is_active', '=', 1)->first();

            if ($expense) {
                DB::beginTransaction();

                $expense->update([
                    'status' => MultiPurposeStatus::REVERSED,
                    'updated_by' => getUserCode(),
                    'updated_at' => getDateTimeNow(),
                ]);

                $modeOfPayment = ExpenseModeOfPayment::where('expense_code', '=', $expenseCode)
                    ->where('is_active', '=', 1)->get();

                if (count($modeOfPayment) > 0) {
                    foreach ($modeOfPayment as $mode_of_payment) {
                        $mode_of_payment->update([
                            'status' => MultiPurposeStatus::REVERSED,
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);

                        $paymentType = $mode_of_payment->payment_type_code;
                        switch ($paymentType) {
                            case PaymentTypeCode::CASH_PAYMENT_CODE:
                                ExpenseCashPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::CARD_PAYMENT_CODE:
                                ExpenseCardPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::CHEQUE_PAYMENT_CODE:
                                ExpenseChequePayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::BANK_PAYMENT_CODE:
                                ExpenseBankPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::ONLINE_TRANSFER_PAYMENT_CODE:
                                ExpenseOnlinePayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'status' => MultiPurposeStatus::REVERSED,
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;
                            default:
                                # code...
                                break;
                        }
                    }
                }


                DB::commit();
                return response()->json([
                    'status' => 200,
                    'message' => $expense
                ]);
            }
             else {
                return response()->json([
                    'status' => 200,
                    'message' => 'Expense Code - ' . $expenseCode . ' Not Found'
                ]);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new Exception($e);
        }
    }

    
    public function update(Request $request, $id)
    {
        try {

            DB::beginTransaction();
            $expenses = Expense::select('id', 'code')->where('code', $id)->first();
            $stockLocation = getCurrentLocationCode($request);
            $expenses->update([
                'expense_type_code' => $request->expenseType['code'],
                'expense_category_code' => $request->expenseCategory['code'],
                //'ledger_type_code'=> $request->ledgerTypeCode,
                'total_amount' => $request->totalAmount,
                'description' => $request->description,
                'date_time' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()
                //'ledger_type_code'=> $request->ledgerTypeCode,                
                // 'payment_type_code' => $request->paymentTypeCode,
                // 'bank_code' => $request->bankCode,
                // 'deposite_date' => $request->depositeDate,
                // 'cheque_date' => $request->chequeDate,
                // 'cheque_number' => $request->chequeNumber,
                // 'ref_number' => $request->refNumber,
                // 'card_number' => $request->cardNumber,
               
            ]);

            if ($expenses) {
                $expensesCode = $request->expense['code'];

                $modeOfPayment = ExpenseModeOfPayment::where('expense_code', '=', $expensesCode)
                    ->where('is_active', '=', 1)->get();

                if (count($modeOfPayment) > 0) {
                    foreach ($modeOfPayment as $mode_of_payment) {
                        $mode_of_payment->update([
                            'payment_type_code' => $mode_of_payment['paymentType'],
                            'amount' => $mode_of_payment['amount'],
                            'updated_by' => getUserCode(),
                            'updated_at' => getDateTimeNow()
                        ]);

                        $paymentType = $mode_of_payment->payment_type_code;
                        switch ($paymentType) {
                            case PaymentTypeCode::CASH_PAYMENT_CODE:
                                ExpenseCashPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'amount' => $mode_of_payment['amount'],
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::CARD_PAYMENT_CODE:
                                ExpenseCardPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'amount' => $mode_of_payment['amount'],
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::CHEQUE_PAYMENT_CODE:
                                ExpenseChequePayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'amount' => $mode_of_payment['amount'],
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::BANK_PAYMENT_CODE:
                                ExpenseBankPayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'amount' => $mode_of_payment['amount'],
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            case PaymentTypeCode::ONLINE_TRANSFER_PAYMENT_CODE:
                                ExpenseOnlinePayment::where('mode_of_payment_code', $mode_of_payment->code)
                                    ->update([
                                        'amount' => $mode_of_payment['amount'],
                                        'updated_by' => getUserCode(),
                                        'updated_at' => getDateTimeNow(),
                                    ]);
                                break;

                            default:
                                # code...
                                break;

                        }

                    }
                }
            }

            DB::commit();
            return response()->json(['status' => 200, 'message' => "Expenses updated"]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $expenses = Expense::select('id', 'is_active')->where('code', $id)->where('is_active', '=', 1)->first();
            $expenses->update([
                'is_active' => 0
            ]);
            return response()->json([
                'status' => 204,
                'message' => "Expenses Types - Deleted"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e
            ], 500);
        }
    }
}