<?php

namespace App\Http\Controllers\Customer_Controllers;

use App\Enums\DefaultValues;
use App\Enums\TransactionCode;
use App\Enums\StockHistorySource;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Customer_Models\SalesQuotation;
use App\Models\Customer_Models\SalesQuotationData;
use App\Models\Stock_Models\StockHistoryTemp;
use Exception;
use App\Models\Item_Models\ItemMaster;
use Illuminate\Support\Facades\DB;

class SalesQuotationController extends Controller
{
    /**
     * index
     * Author: Suhail Jamaldeen
     * Date: 05.01.2022
     * Version: 01
     * Logic: Get the Sales Quotations based for the current location
     * @param  mixed $request
     * @return void
     */
    public function index(Request $request)
    {
        try {

            $salesQuotations = SalesQuotation::select(
                'code',
                'customer_code',
                'stock_location_code',
                'date',
                'prints',
                'is_locked',
                'total_amount'
            )
                ->with([
                    'StockHistoryTemps' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'customer.route.salesRep' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])

                ->where('stock_location_code', '=', getCurrentLocationCode($request))
                ->where('is_active', '=', 1)
                ->get();
            return response()->Json(['status' => 200, 'salesQuotations' => $salesQuotations]);

        } catch (\Exception $e) {
            return response(['status' => 500, 'Message' => $e]);
            throw new Exception($e);
        }
    }

    /**
     * show
     * Author: Suhail Jamaldeen
     * Version: 01
     * Date: 11.1.2023
     * Logic: Display the sales quotation for id.
     * @param  mixed $id
     * @return void
     */
    public function show($id)
    {
        try {
            $salesQuotation = SalesQuotation::select(
                'code',
                'stock_location_code',
                'customer_code',
                'date',
                'discount_amount',
                'discount_percentage',
                'stock_location_code',
                'quotation_discount',
                'sub_total_amount',
                'remarks',
                'total_amount',
                'prints',
                'is_locked'
            )
                ->with([
                    'StockHistoryTemps' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'StockHistoryTemps.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'customer.route.salesRep' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('code', $id)
                ->where('is_active', '=', 1)
                ->first();

            return response()->Json([
                'status' => 200,
                'salesQuotation' => $salesQuotation
            ]);
        } catch (\Exception $e) {
            return response(
                ['status' => 500, 'Message' => $e]
            );
        }
    }

    public function create()
    {
        //
    }


    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $sales = TransactionCode::SALES_QUOTATION;
            $stockLocationCode = getCurrentLocationCode($request);
            $locationPrefix = substr("$stockLocationCode", -2);
            $max_code = SalesQuotation::select('code')
                ->where('stock_location_code', '=', $stockLocationCode)
                ->max('code');

            $max_id = $max_code == null || $max_code == '' ? config('global.code_value') + 1 : substr("$max_code", 5) + 1;

            $salesQuotation = SalesQuotation::create([
                'code' => $sales . $locationPrefix . $max_id,
                'customer_code' => $request['customer']['code'],
                'stock_location_code' => getCurrentLocationCode($request),
                'date' => $request->date,
                'sub_total_amount' => $request->subTotalAmount,
                'discount_amount' => $request->discountAmount,
                'discount_percentage' => $request->discountPercentage,
                'quotation_discount' => $request->quotationDiscount,
                'total_amount' => $request->totalAmount,
                'prints' => 1,
                'is_locked' => $request->isLocked,
                'remarks' => $request->remarks,
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()


            ]);
            if ($salesQuotation) {
                $stockHistoryTemp = $request->stockHistoryTemps;
                if (count($stockHistoryTemp) > 0) {
                    foreach ($stockHistoryTemp as $stockHistoryTemp) {
                        $this->addStockHistoryTemp($stockHistoryTemp, $salesQuotation->code, StockHistorySource::SALES_QUOTATION);
                    }
                }
            }


            DB::commit();
            return response(['status' => 200, 'Message' => 'Sales Quotation Created']);
        } catch (\Exception $e) {
            DB::rollback();
            return response(['status' => 500, 'Message' => $e]);
        }
    }

    private function addStockHistoryTemp($stockHistoryTemp, $sourceCode, $source)
    {
        try {
            DB::beginTransaction();
            $order = TransactionCode::STOCK_HISTORY_TEMP;
            $max_code = StockHistoryTemp::max('code');
            $max_id = $max_code == null ? config('global.code_value') + 1 : substr("$max_code", 3) + 1;

            $stockHistoryTemp = StockHistoryTemp::create([
                'code' => $order . $max_id,
                'item_code' => $stockHistoryTemp['itemMaster']['code'] ?? $stockHistoryTemp['itemCode'],
                'source_code' => $sourceCode,
                'stock_code' => $stockHistoryTemp['stockCode'],
                'source' => $source,
                'wholesale_price' => $stockHistoryTemp['wholesalePrice'] ?? null,
                'retail_price' => $stockHistoryTemp['retailPrice'] ?? null,
                'cost_price' => $stockHistoryTemp['costPrice'] ?? null,
                'quantity' => $stockHistoryTemp['quantity'] ?? null,
                'remaining_quantity' => null,
                'sub_total_amount' => $stockHistoryTemp['subTotalAmount'],
                'discount_amount' => $stockHistoryTemp['discountAmount'] ?? null,
                'discount_percentage' => $stockHistoryTemp['discountPercentage'] ?? null,
                'discount_total' => $stockHistoryTemp['discountTotal'] ?? null,
                'total_amount' => $stockHistoryTemp['totalAmount'],
                'created_by' => getUserCode(),
                'created_at' => getDateTimeNow(),
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow()

            ]);


            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            throw new Exception($e);
        }
    }


    public function edit($id)
    {
        //
    }


    /**
     * update
     * Logic: when update particular quotation, delete all items for particular quotaion in stockhistorytemp and newly add that items.
     * version : 1
     * Date :2023.01.15
     * @param  mixed $request
     * @param  mixed $id
     * @return void
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $salesQuotation = SalesQuotation::where('code', $id)
            ->first();

            $salesQuotation->update([
                'customer_code' => $request['customer']['code'],
                'date' => $request->date,
                'sub_total_amount' => $request->subTotalAmount,
                'discount_amount' => $request->discountAmount,
                'discount_percentage' => $request->discountPercentage,
                'quotation_discount' => $request->quotationDiscount,
                'total_amount' => $request->totalAmount,
                'prints' => 1,
                'is_locked' => $request->isLocked,
                'remarks' => $request->remarks,
                'updated_by' => getUserCode(),
                'updated_at' => getDateTimeNow(),

            ]);
            StockHistoryTemp::where('source_code', $salesQuotation->code)
            ->delete();

            if ($salesQuotation) {
                $stockHistoryTemp = $request->stockHistoryTemps;
                if (count($stockHistoryTemp) > 0) {
                    foreach ($stockHistoryTemp as $stockHistoryTemp) {
                        $this->addStockHistoryTemp($stockHistoryTemp, $salesQuotation->code, StockHistorySource::SALES_QUOTATION);
                    }
                }
            }


            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return response(['status' => 500, 'Message' => $e]);
        }
    }


    public function getSalesQuotationByCustomer(Request $request)
    {
        try {
            $customerCode = $request->key;
            $salesQuotation = SalesQuotation::select(
                'code',
                'customer_code',
                'date',
                'discount_amount',
                'discount_percentage',
                'stock_location_code',
                'quotation_discount',
                'sub_total_amount',
                'remarks',
                'total_amount',
                'prints',
                'is_locked'
            )
                ->with([
                    'StockHistoryTemps' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'StockHistoryTemps.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('customer_code', $customerCode)
                ->where('is_active', '=', 1)
                ->get();

            return response()->Json([
                'status' => 200,
                'salesQuotation' => $salesQuotation
            ]);
        } catch (\Exception $e) {
            return response(
                ['status' => 500, 'Message' => $e]
            );
        }
    }
    /**
     * salesQuotationSearch
     * Author: Suhail Jamaldeen
     * Date: 14.01.2023
     * Version: 01
     * Logic: Search the quotations by code
     * @param  mixed $request
     * @return void
     */
    public function salesQuotationSearch($key)
    {
        try {
            $salesQuotation = SalesQuotation::select(
                'code',
                'stock_location_code',
                'customer_code',
                'date',
                'discount_amount',
                'discount_percentage',
                'stock_location_code',
                'quotation_discount',
                'sub_total_amount',
                'remarks',
                'total_amount',
                'prints',
                'is_locked',
                'code AS name',
            )
                ->with([
                    'StockHistoryTemps' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'StockHistoryTemps.itemMaster' => function ($query) {
                        $query->where('is_active', '=', 1);
                    },
                    'StockHistoryTemps.itemMaster.itemUnit' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'customer.customerAddresses' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->with([
                    'stockLocation' => function ($query) {
                        $query->where('is_active', '=', 1);
                    }
                ])
                ->where('code', 'like', "%$key%")
                ->where('is_active', '=', 1)
                ->get();

            return response()->Json([
                'status' => 200,
                'salesQuotation' => $salesQuotation
            ]);
        } catch (\Exception $e) {
            return response(
                ['status' => 500, 'Message' => $e]
            );
        }
    }


    public function itemSearchForQuotation(Request $request)
    {
        $key = $request->key;
        try {
            $item = ItemMaster::select(
                'code',
                'reference_code',
                'barcode',
                'name',
                'description',
                'category_code',
                'sub_category_code',
                'unit_code',
                'brand_code',
                'color_code',
                'bin_location_code',
                'wholesale_price',
                'retail_price',
                'dealer_price',
                'least_price',
                'cost_price',
                'model',
                'min_stock',
                'max_stock',
                'reorder_stock',
                'expiry_notification_days'
            )

                ->where([['is_active', '=', 1]])
                ->where(function ($query) use ($key) {
                    $query->Where('code', 'like', "%$key%")
                        ->orWhere('description', 'like', "%$key%")
                        ->orWhere('name', 'like', "%$key%")
                        ->orWhere('barcode', 'like', "%$key%");
                })->get();

            return response()->json(['status' => 200, 'item' => $item]);
        } catch (\Exception $e) {
            throw new Exception($e);
        }
    }

    public function destroy($id)
    {
        //
    }
}
